#!/usr/bin/tclsh
#* 
#* ------------------------------------------------------------------
#* GoogleAdsenseChannelsBarGraph.tcl - Channel Bar Charts
#* Created by Robert Heller on Sat Sep  8 11:17:43 2007
#* ------------------------------------------------------------------
#* Modification History: $Log: headerfile.text,v $
#* Modification History: Revision 1.1  2002/07/28 14:03:50  heller
#* Modification History: Add it copyright notice headers
#* Modification History:
#* ------------------------------------------------------------------
#* Contents:
#* ------------------------------------------------------------------
#*  
#*     Generic Project
#*     Copyright (C) 2005  Robert Heller D/B/A Deepwoods Software
#* 			51 Locke Hill Road
#* 			Wendell, MA 01379-9728
#* 
#*     This program is free software; you can redistribute it and/or modify
#*     it under the terms of the GNU General Public License as published by
#*     the Free Software Foundation; either version 2 of the License, or
#*     (at your option) any later version.
#* 
#*     This program is distributed in the hope that it will be useful,
#*     but WITHOUT ANY WARRANTY; without even the implied warranty of
#*     MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
#*     GNU General Public License for more details.
#* 
#*     You should have received a copy of the GNU General Public License
#*     along with this program; if not, write to the Free Software
#*     Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
#* 
#*  
#* 

set argv0 [file rootname [info script]]

package require Tk
package require BWidget
package require BWHelp;#                BWidget help package
package require BWStdMenuBar;#          BWidget menu bar package
package require csv
package require struct::matrix
package require snit

# Help Directory setup
global HelpDir
set HelpDir [file join [file dirname [file dirname [file dirname \
                                                        [info script]]]] Help]
namespace eval ChannelGraph {
  variable Status {}
  variable Main [MainFrame::create .mainframe \
	-menu [StdMenuBar::MakeMenu -file {"&File" {file} {file:menu} 0 {
		{command "&New" {file:new} "Clear chart" {Ctrl n} \
				-command ChannelGraph::Clear}
		{command "&Open..." {file:open} "Open a CSV file" {Ctrl o} \
				-command ChannelGraph::Open}
		{command "&Save" {file:save} "Save Postscript of chart" {Ctrl s} \
				-command ChannelGraph::Save}
		{command "Save &As.." {file:saveas} "Save Postscript of chart" {Ctrl a} \
				-command ChannelGraph::SaveAs}
		{command "&Print..." {file:print} "Print chart" {Ctrl p} \
				-command ChannelGraph::Print}
		{command "&Close" {file:close} "Close the application" {Ctrl q} -command ::exit}
		{command "E&xit"  {file:exit}  "Exit the application" {Ctrl q} -command ::exit}
		}} -options {"&Options" {options} {options:menu} 0 {
		{command "Select Value Column to Graph" {options:selval} 
			 "Select Value Column to Graph" {} 
			 -command ChannelGraph::SelColumn}
		}}] \
	-textvariable ChannelGraph::Status]
  pack $Main -expand yes -fill both
  $Main showstatusbar status
  set frame [$Main getframe]
  set sw [ScrolledWindow::create $frame.sw -scrollbar both -auto both]
  pack $sw -expand yes -fill both
  variable Chart [canvas [$sw getframe].chart -background white]
  pack $Chart -expand yes -fill both
  $sw setwidget $Chart


  variable XTickFormat %4.1f
  variable ChartLabelFont [font create -family Courier -size 12 -weight bold -slant roman]
  variable ChartLabelFontLarge [font create -family Times -size 24 -weight bold -slant roman]
  variable PointLabelWidth [font measure $ChartLabelFont -displayof $Chart [format $XTickFormat 99.9]]
  variable YTickLabelWidth 

  variable ChartWidth [expr {$PointLabelWidth * 11.0}]
  variable ChartHeight

  variable PageHeight 10i
  variable PageWidth   8i
  variable Landscape  yes

  variable CSVMatrix
  variable DataArray
  variable CSVFilename {}
  variable GraphValueColumn {Page CTR}
  variable GraphValueConversionProc ChannelGraph::StripPercent

  variable GraphValueColumnStripProcs
  array set GraphValueColumnStripProcs {
    {Page impressions} ChannelGraph::StripNothing
    Clicks ChannelGraph::StripNothing
    {Page CTR} ChannelGraph::StripPercent
    {Page eCPM} ChannelGraph::StripNothing
    Earnings ChannelGraph::StripNothing
  }

  variable CSVFileTypes {
    {{CSV Files} {.csv} TEXT}
    {{All Files} *      TEXT}
  }
  variable PSFileTypes {
    {{PostSript Files} {.ps} TEXT}
    {{All Files} *      TEXT}
  }
  variable Colors {black brown red {dark orange} green blue {dark violet} {dark gray}}

  snit::type SelectColumnDialog {
    pragma -hastypedestroy no
    pragma -hasinstances no
    pragma -hastypeinfo no

    typecomponent dialog
    typecomponent columnsCB

    typeconstructor {
      set dialog [Dialog::create .selectColumnDialog -bitmap questhead \
			-default 0 -cancel 1 -modal local -transient yes \
			-parent . -side bottom \
			-title {Select a column to graph}]
      $dialog add -name ok -text OK -command [mytypemethod _OK]
      $dialog add -name cancel -text Cancel -command [mytypemethod _Cancel]
      $dialog add -name help -text Help -command [list BWHelp::HelpTopic {Select a column to graph dialog}]
      set frame [$dialog getframe]
      set lf [LabelFrame::create $frame.columnsCBLF -text "Data Column to graph:" -side left]
      pack $lf -fill x
      set lfframe [$lf getframe]
      set values [lsort -dictionary \
			[array names \
				::ChannelGraph::GraphValueColumnStripProcs]]
      set columnsCB [ComboBox::create $lfframe.columnsCB \
	-editable no \
	-values $values]
      $columnsCB setvalue @[lsearch -exact $values "$::ChannelGraph::GraphValueColumn"]
      pack $columnsCB -fill x -expand yes
    }
    typemethod _OK {} {
      set answer "[$columnsCB cget -text]"
      Dialog::withdraw $dialog
      return [Dialog::enddialog $dialog "$answer"]
    }
    typemethod _Cancel {} {
      Dialog::withdraw $dialog
      return [Dialog::enddialog $dialog {}]
    }
    typemethod draw {args} {
      set parent [from args -parent .]
      $dialog configure -parent $parent
      wm transient [winfo toplevel $dialog] $parent
      return [Dialog::draw $dialog]
    }
  }
}

proc ChannelGraph::StripPercent {rawvalue} {
  regsub {%$} "$rawvalue" {} value
  return $value
}

proc ChannelGraph::StripDollar {rawvalue} {
  regsub {^\$} "$rawvalue" {} value
  return $value
}

proc ChannelGraph::StripNothing {value} {return $value}

proc ChannelGraph::Clear {} {
  variable Chart
  variable CSVMatrix
  variable CSVFilename {}

  $Chart delete all
  catch {$CSVMatrix destroy}
  catch {unset CSVMatrix}
}

proc ChannelGraph::Open {{filename {}}} {
  variable Main
  variable Chart
  variable ChartHeight
  variable ChartWidth
  variable XTickFormat
  variable ChartLabelFont
  variable ChartLabelFontLarge
  variable PointLabelWidth
  variable YTickLabelWidth
  variable CSVMatrix
  variable DataArray
  variable CSVFilename
  variable CSVFileTypes
  variable Colors
  variable GraphValueColumn
  variable GraphValueConversionProc
  variable GraphValueColumnStripProcs

  if {[string equal "$filename" {}]} {
    set filename [tk_getOpenFile -initialdir [pwd] -initialfile {} \
				 -filetypes $CSVFileTypes \
				 -title "File to open" \
				 -parent $Main]
  }
  if {[string equal "$filename" {}]} {return}
  if {[catch {open "$filename" r} fp]} {
    tk_messageBox -type ok -icon error -message "Could not open $filename: $fp"
    return
  }
  fconfigure $fp -encoding unicode
  set CSVMatrix [::struct::matrix]
  array unset DataArray
  ::csv::read2matrix $fp $CSVMatrix "\t" auto
  close $fp
#  puts stderr "*** ChannelGraph::Open: $CSVMatrix cells = [$CSVMatrix cells], [$CSVMatrix rows]x[$CSVMatrix columns]"  
  set CSVFilename $filename

  $Chart delete all

  set channels {}

  set peakValue 0.0  
  set YTickLabelWidth 0

  for {set irow 1} {$irow < [expr {[$CSVMatrix rows] - 2}]} {incr irow} {
    set thisChannel "[$CSVMatrix get cell 0 $irow]"
    for {set icol 0} {$icol < [$CSVMatrix columns]} {incr icol} {
      set slot "[$CSVMatrix get cell 0 $irow],[$CSVMatrix get cell $icol 0]"
      set DataArray("$slot") "[$CSVMatrix get cell $icol $irow]"
      if {[string equal "[$CSVMatrix get cell $icol 0]" "$GraphValueColumn"]} {
	set value [$GraphValueConversionProc "[$CSVMatrix get cell $icol $irow]"]
	if {$value == 0.0} {continue}
	if {$value > $peakValue} {set peakValue $value}
#	puts "*** ChannelGraph::Open: slot = $slot, value = $value"
	if {[lsearch -exact $channels "$thisChannel"] < 0} {
	  lappend channels "$thisChannel"
	  set ytw  [font measure $ChartLabelFont -displayof $Chart " $thisChannel "]
	  if {$ytw > $YTickLabelWidth} {set YTickLabelWidth $ytw}
#	  puts "*** ChannelGraph::Open: channels = $channels"
	}
      }
    }
  }
#  parray DataArray
#  puts stderr "*** ChannelGraph::Open: channels = $channels"
#  puts "*** ChannelGraph::Open: peakValue = $peakValue"

  set channels [lsort -dictionary $channels]

  set bh [expr {[font metrics $ChartLabelFont -displayof $Chart -linespace] * 1.1}]
  set bh_2 [expr {$bh / 2.0}]
  set ChartHeight [expr {$bh * [llength $channels]}]

  set togValue [RoundUpPeak $peakValue]
  
  set PointLabelWidth [font measure $ChartLabelFont -displayof $Chart [format $XTickFormat $togValue]]
  set ChartWidth [expr {$PointLabelWidth * 11.0}]

#  puts "*** ChannelGraph::Open: togValue = $togValue"

  set xTick [expr {$togValue / 10.0}]
  set xScale [expr {$ChartWidth / double($togValue)}]

#  puts "*** ChannelGraph::Open: yTick = $yTick"

  $Chart create text $ChartWidth 0 \
		-text "$GraphValueColumn" -font $ChartLabelFontLarge \
		-anchor se

  $Chart create line 0 $ChartHeight $ChartWidth $ChartHeight
  $Chart create line 0 0 0 $ChartHeight

  

  set Y [expr {$bh / 2.0}]
  foreach channel $channels {
    $Chart create line 0 $Y -2.5 $Y
    $Chart create text -5.0 $Y -text "$channel" -font $ChartLabelFont -anchor e
    set Y [expr {$Y + $bh}]
  }

  for {set tick 0.0} {$tick <= $togValue} {set tick [expr {$tick + $xTick}]} {
    set X [expr {$tick * $xScale}]
    $Chart create line $X $ChartHeight $X [expr {$ChartHeight + 2.5}]
    $Chart create text $X [expr {$ChartHeight + 5.0}]  -text [format $XTickFormat $tick] -font $ChartLabelFont -anchor n
  }

  set legendY [expr {$ChartHeight + 20.0}]

  set colors {}
  set Y [expr {$bh / 2.0}]
  foreach chan $channels {
    if {[llength $colors] == 0} {set colors $Colors}
    if {[catch {$GraphValueConversionProc $DataArray("$chan,$GraphValueColumn")} value]} {
	set value 0.0
    }
    set chancolor [lindex $colors 0]
    set colors [lrange $colors 1 end]
    $Chart create rect 0.0 $legendY 10 [expr {$legendY + 10}] -outline black -fill $chancolor
    $Chart create text 12.5 $legendY -text "$chan: $value" -font $ChartLabelFont -anchor nw -fill $chancolor
    set legendY [expr {$legendY + 20}]
    set bw [expr {$value * $xScale}]
    $Chart create rect 0 [expr {$Y - $bh_2}] \
		       $bw [expr {$Y + $bh_2}] -outline {} -fill $chancolor
    set Y [expr {$Y + $bh}]    
  }

  $Chart configure -scrollregion [$Chart bbox all]
  $Chart xview moveto 0.0
  $Chart yview moveto 0.0
}

proc ChannelGraph::RoundUpPeak {value} {
  if {$value < 1} {
    return 1.0
  } else {
    set p10 [expr {int(log10($value))}]
    set mult10 [expr {pow(10,$p10)}]
    return [expr {ceil($value / double($mult10)) * $mult10}]
  }
}

proc ChannelGraph::Save {} {
  variable CSVFilename
  SaveAs "[file rootname $CSVFilename].ps"
}

proc ChannelGraph::SaveAs {{filename {}}} {
  variable PSFileTypes
  variable CSVFilename
  variable Main

  if {[string equal "$filename" {}]} {
    set filename [tk_getSaveFile -initialdir [file dirname $CSVFilename] \
				 -initialfile "[file rootname $CSVFilename].ps" \
				 -defaultextension .ps \
				 -filetypes $PSFileTypes \
				 -parent $Main \
				 -title "PostScript file to save chart as"]
  }
  if {[string equal "$filename" {}]} {return}
  if {[catch {open  "$filename" w} fp]} {
    tk_messageBox -type ok -icon error -message "Could not open $filename: $fp"
    return
  }
  WriteChartPS $fp
  close $fp
}

proc ChannelGraph::Print {} {
  variable PageHeight
  variable PageWidth
  variable Landscape
  variable Main

  set fp [Printer::OpenPrinter "Print Chart" $Main PageHeight PageWidth Landscape]
  if {[string equal "$fp" {}]} {return}
  WriteChartPS $fp
  catch {close $fp} message
  tk_messageBox -type ok -icon info -message "$message"
}

proc ChannelGraph::WriteChartPS {pschan} {
  variable PageHeight
  variable PageWidth
  variable Landscape
  variable Chart

  set sr [$Chart cget -scrollregion]
  set x [lindex $sr 0]
  set y [lindex $sr 1]
  set width [expr {[lindex $sr 2] - $x + 1}]
  set height [expr {[lindex $sr 3] - $y + 1}]
  $Chart postscript -channel $pschan -height $height -width $width \
			-x $x -y $y  -rotate $Landscape \
			-pageheight $PageHeight -pagewidth $PageWidth
}

proc ChannelGraph::SelColumn {} {
  variable GraphValueColumn
  variable GraphValueConversionProc
  variable GraphValueColumnStripProcs
  variable Main
  variable CSVFilename

  set result [SelectColumnDialog draw -parent $Main]
  if {[string length "$result"] == 0} {return}
  set GraphValueColumn "$result"
  set GraphValueConversionProc "$GraphValueColumnStripProcs($result)"
  if {[string length "$CSVFilename"] == 0} {return}
  Open "$CSVFilename"
}

namespace eval Printer {
  snit::type PrinterDialog {
    pragma -hastypeinfo    no
    pragma -hastypedestroy no
    pragma -hasinstances   no

    typecomponent dialog;#			Dialog widget
    typecomponent   printerOrFileTF;#		Printer or file title frame
    typecomponent     printerRadio;#		printer radiobutton
    typecomponent     printerLF;#		printer label frame
    typecomponent       printerCB;#		printer ComboBox
    typecomponent     fileRadio;#		file radiobutton
    typecomponent     fileLF;#			file label frame
    typecomponent       fileE;#			file entry
    typecomponent       fileB;#			file browse button
    typecomponent   landscapeCheck;#		landscape mode checkbutton
    typecomponent   pageSizeTF;#		pagesize title frame
    typecomponent     pageWidthLF;#		page width label frame
    typecomponent       pageWidthSB;#		page width spinbox
    typecomponent	pageWidthUnitsCB;#	page width units ComboBox
    typecomponent     pageHeightLF;#		page height label frame
    typecomponent       pageHeightSB;#		page width spinbox
    typecomponent       pageHeightUnitsCB;#	page height units ComboBox

    typevariable _Options -array {
      -title "Print Dialog"
      -parent .
      -pageheight 10i
      -pagewidth   8i
      -landscape   no
      -printerpath {}      
    }
    typevariable _phUnits i
    typevariable _pwUnits i
    typevariable _printerOrFile printer
    typeconstructor {
      set dialog [Dialog::create .printerDialog -anchor c -bitmap questhead \
				-cancel 1 -default 0 -modal local \
				-parent $_Options(-parent) -side bottom \
				-title "$_Options(-title)" -transient yes]
      $dialog add -name print  -text "Print"  -command [mytypemethod _Print]
      $dialog add -name cancel -text "Cancel" -command [mytypemethod _Cancel]
      $dialog add -name help   -text "Help"   -command "BWHelp::HelpTopic {Print Dialog}"
      set dframe [$dialog getframe]
      set printerOrFileTF [TitleFrame::create $dframe.printerOrFileTF \
				-text "Printer or File" \
				-side left]
      pack $printerOrFileTF -fill both -expand yes
      set printerOrFileTFfr [$printerOrFileTF getframe]
      set printerRadio [radiobutton $printerOrFileTFfr.printerRadio \
				-command [mytypemethod _TogglePF] \
				-indicatoron yes -value printer -text {} \
				-variable [mytypevar _printerOrFile]]
      grid $printerRadio -row 0 -column 0 -sticky wns
      set printerLF [LabelFrame::create $printerOrFileTFfr.printerLF \
				-text "Printer:" -width 10]
      grid $printerLF -row 0 -column 1 -sticky news
      set printerCB [ComboBox::create [$printerLF getframe].printerCB \
			-editable no]
      pack $printerCB -fill x -expand yes
      set fileRadio [radiobutton $printerOrFileTFfr.fileRadio \
				-command [mytypemethod _TogglePF] \
				-indicatoron yes -value file -text {} \
				-variable [mytypevar _printerOrFile]]
      grid $fileRadio -row 1 -column 0 -sticky wns
      set fileLF [LabelFrame::create $printerOrFileTFfr.fileLF \
				-text "File:" -width 10 -state disabled]
      set fileE [Entry::create [$fileLF getframe].fileE -state disabled \
				-text "printout.ps"]
      pack $fileE -expand yes -fill x -side left
      set fileB [Button::create [$fileLF getframe].fileB -text "Browse" \
		 	-command [mytypemethod _FileBrowsePS] -state disabled]
      pack $fileB -side right
      grid $fileLF -row 1 -column 1 -sticky news
      set landscapeCheck [checkbutton $dframe.landscapeCheck \
				-text "Landscape mode?" \
				-indicatoron yes -onvalue yes -offvalue no \
				-variable [mytypevar _Options(-landscape)]]
      pack $landscapeCheck -expand yes -fill x
      set pageSizeTF [TitleFrame::create $dframe.pageSizeTF \
			-text "Page Size" -side left]
      pack $pageSizeTF -fill both -expand yes
      set pageSizeTFfr [$pageSizeTF getframe]
      set pageWidthLF [LabelFrame::create $pageSizeTFfr.pageWidthLF \
			-text "Width:" -width 15]
      pack $pageWidthLF -fill x
      set pageWidthSB [SpinBox::create [$pageWidthLF getframe].pageWidthSB \
			-range {1.0 3000.0 1.0} -editable yes]
      pack $pageWidthSB -expand yes -fill x -side left
      set pageWidthUnitsCB [ComboBox::create \
			[$pageWidthLF getframe].pageWidthUnitsCB \
			-values {c i m p} -editable no -width 1 \
			-textvariable _pwUnits]
      pack $pageWidthUnitsCB -side right
      set pageHeightLF [LabelFrame::create $pageSizeTFfr.pageHeightLF \
			-text "Height:" -width 15]
      pack $pageHeightLF -fill x
      set pageHeightSB [SpinBox::create [$pageHeightLF getframe].pageHeightSB \
			-range {1.0 3000.0 1.0} -editable yes]
      pack $pageHeightSB -expand yes -fill x -side left
      set pageHeightUnitsCB [ComboBox::create \
			[$pageHeightLF getframe].pageHeightUnitsCB \
			-values {c i m p} -editable no -width 1 \
			-textvariable _phUnits]
      pack $pageHeightUnitsCB -side right
    }
    typemethod _TogglePF {} {
      switch $_printerOrFile {
	printer {
	  $printerLF configure -state normal
	  $printerCB configure -state normal
	  $fileLF    configure -state disabled
	  $fileE     configure -state disabled
	  $fileB     configure -state disabled
	}
	file {
	  $printerLF configure -state disabled
	  $printerCB configure -state disabled
	  $fileLF    configure -state normal
	  $fileE     configure -state normal
	  $fileB     configure -state normal
	}
      }
    }
    typemethod _FileBrowsePS {} {
      set file "[$fileE cget -text]"
      set newfile [tk_getSaveFile \
	-filetypes { {{Postscript Files} .ps TEXT} {{All Files} * TEXT}}\
	-defaultextension .ps -initialdir "[file dirname $file]" \
	-initialfile "$file" -parent $dialog \
	-title "Postscript file to print to"]
      if {[string length "$newfile"] > 0} {
	$fileE configure -text "$newfile"
      }
    }
    typemethod _Cancel {} {
      $dialog withdraw
      $dialog enddialog cancel
    }
    typemethod _Print {} {
      switch $_printerOrFile {
	printer {
	  set lp [auto_execok lp]
	  set lpr [auto_execok lpr]
	  if {[string length "$lpr"] > 0} {
	    set _Options(-printerpath) "|$lpr -P[$printerCB cget -text]"
	  } elseif {[string length "$lp"] > 0} {
	    set _Options(-printerpath) "|$lp -d [$printerCB cget -text]"
	  } else {
	    tk_messageBox -type ok -icon warning "Print spool command not available, please print to a file."
	    return
  	  }
	}
	file {
	  set _Options(-printerpath) "[$fileE cget -text]"
	}
      }
      set _Options(-pageheight) "[$pageHeightSB cget -text][$pageHeightUnitsCB cget -text]"
      set _Options(-pagewidth) "[$pageWidthSB cget -text][$pageWidthUnitsCB cget -text]"
      $dialog withdraw
      $dialog enddialog print
    }
    typemethod cget {option} {
      return $_Options($option)
    }
    typemethod draw {args} {
#      puts stderr "$type draw $args"
      foreach o [array names _Options] {
	set _Options($o) [from args $o $_Options($o)]
      }
#      parray _Options
      $dialog configure -parent $_Options(-parent)
      $dialog configure -title  "$_Options(-title)"
      wm transient [winfo toplevel $dialog] $_Options(-parent)
      set ph 10
      regexp {^([0-9.-]+)([cimp]*)$} "$_Options(-pageheight)" -> ph _phUnits
      $pageHeightSB configure -text "$ph"
      switch [string length "$_phUnits"] {
        0 {set _phUnits p}
	1 {}
	default {
	  set _phUnits [string range "$_phUnits" 0 0]
	}
      }
      $pageHeightUnitsCB configure -text "$_phUnits"
      set pw 8
      regexp {^([0-9.-]+)([cimp]*)$} "$_Options(-pagewidth)" -> pw _pwUnits
      $pageWidthSB configure -text "$pw"
      switch [string length "$_pwUnits"] {
        0 {set _pwUnits p}
	1 {}
	default {
	  set _pwUnits [string range "$_pwUnits" 0 0]
	}
      }
      $pageWidthUnitsCB configure -text "$_pwUnits"
      set lpstat [auto_execok lpstat]
      if {[string length "$lpstat"] > 0} {
	set printers {}
	if {![catch {open "|$lpstat -a" r} lpstatFP]} {
	  while {[gets $lpstatFP line] >= 0} {
	    if {[regexp {^(.+)[[:space:]]accepting} "$line" -> printer] > 0} {
	      lappend printers $printer
	    }
	  }
	  close $lpstatFP
	}
	$printerCB configure -values [lsort -dictionary $printers]
	if {![catch {open "|$lpstat -d" r} lpstatFP]} {
	  set defprinter {}
	  while {[gets $lpstatFP line] >= 0} {
	    if {[regexp {destination:[[:space:]](.*)$} "$line" -> defprinter] > 0} {
	      break
	    }
	  }
	  close $lpstatFP
	  $printerCB configure -text "$defprinter"
	}
      }
      return [$dialog draw]
    }
  }
}

proc Printer::OpenPrinter {title parent pageheightvar pagewidthvar 
			   landscapevar} {
  upvar $pageheightvar pageheight
  upvar $pagewidthvar  pagewidth
  upvar $landscapevar  landscape

  set button [Printer::PrinterDialog draw -title "$title" -parent $parent \
					  -pageheight $pageheight \
					  -pagewidth $pagewidth \
					  -landscape $landscape]
  switch $button {
    cancel {return {}}
    print  {
      set pageheight [Printer::PrinterDialog cget -pageheight]
      set pagewidth  [Printer::PrinterDialog cget -pagewidth]
      set landscape  [Printer::PrinterDialog cget -landscape]
      set printerpath [Printer::PrinterDialog cget -printerpath]
      if {[catch {open "$printerpath" w} prfp]} {
	tk_messageBox -type ok -icon error -message "$parent: Printer::OpenPrinter: $prfp"
        return {}
      }
      return $prfp
    }
  }
}


