#* 
#* ------------------------------------------------------------------
#* DatabaseWidgets.tcl - Database Widgets
#* Created by Robert Heller on Thu May 24 09:43:36 2007
#* ------------------------------------------------------------------
#* Modification History: $Log: headerfile.text,v $
#* Modification History: Revision 1.1  2002/07/28 14:03:50  heller
#* Modification History: Add it copyright notice headers
#* Modification History:
#* ------------------------------------------------------------------
#* Contents:
#* ------------------------------------------------------------------
#*  
#*     Generic Project
#*     Copyright (C) 2005  Robert Heller D/B/A Deepwoods Software
#* 			51 Locke Hill Road
#* 			Wendell, MA 01379-9728
#* 
#*     This program is free software; you can redistribute it and/or modify
#*     it under the terms of the GNU General Public License as published by
#*     the Free Software Foundation; either version 2 of the License, or
#*     (at your option) any later version.
#* 
#*     This program is distributed in the hope that it will be useful,
#*     but WITHOUT ANY WARRANTY; without even the implied warranty of
#*     MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
#*     GNU General Public License for more details.
#* 
#*     You should have received a copy of the GNU General Public License
#*     along with this program; if not, write to the Free Software
#*     Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
#* 
#*  
#* 

#**********************************************************************
#*                                                                    *
#* Database Widgets.  Various widgets and things that interface to the*
#* database.
#*                                                                    *
#**********************************************************************

package require snit;#		Snit package
package require BWidget;#	BWidget package
package require snodbc;#	snodbc (database) package
package require Printer;#	Printer package
package require ReportPages;#	Report Pages widget
package require csv;#		CSV Package

namespace eval DatabaseWidgets {
  # Constant information about the address database
  variable TableName triketrips
  # Table specification
  variable TableSpecification {(starttime timestamp not null,
	endtime   timestamp not null,
	distance real not null,
	maxspeed real not null,
	duration integer not null,
	averagespeed real not null,
	comments text)}
  # Dummy type to generate unique ids
  snit::type statementID {
    pragma -hasinstances false
    typevariable id 0
    typemethod create {} {
      incr id
      return "${type}${id}"
    }
  }
  variable LabelWidth 15;# Standard label width
  #**********************************
  # Add trip frame
  #**********************************
  snit::widget AddFrame {
    variable _Connection {};#		Database connection object
    variable _InsertTrip {};#		Insert statement object

    # Frame components
    component startLF;#			Start time of trip
    component   startMonthSB;#		  Month
    component   startDaySB;#		  Day
    component   startYearSB;#		  Year
    component   startHrSB;#		  Hour
    component   startMinSB;#		  Minutes
    component endLF;#			End time of trip
    component   endMonthSB;#		  Month
    component   endDaySB;#		  Day
    component   endYearSB;#		  Year
    component   endHrSB;#		  Hour
    component   endMinSB;#		  Minutes
    component distanceLF;#		Trip distance
    component   distMiSB;#		  Miles
    component maxspeedLF;#		Max speed.
    component   maxMPHSB;#		  MPH
    component commentsTF;#		Comments
    component   commentsText
    component button;#                  Add Trip Button

    #***************************************
    # Constructor: populate frame with GUI elememnts.
    #***************************************
    constructor {args} {
      # Start time label frame. Contains the hour and minute of the start time.
      install startLF using LabelFrame::create $win.startLF \
				-text "Start time:" \
				-width $::DatabaseWidgets::LabelWidth \
				-state disabled
      pack $startLF -fill x
      set startLFfr [$startLF getframe]
      install startMonthSB using SpinBox::create $startLFfr.startMonthSB \
			-width 2 \
			-range {1 12 1} -editable yes -state disabled \
			-modifycmd [mymethod _updateDays start]
      $startMonthSB setvalue first
      pack $startMonthSB -side left
      pack [Label::create $startLFfr.s1 -text {/}] -side left
      install startDaySB using SpinBox::create $startLFfr.startDaySB \
      			-width 2 \
			-range {1 31 1} -editable yes -state disabled
      $startDaySB setvalue first
      pack $startDaySB -side left
      pack [Label::create $startLFfr.s2 -text {/}] -side left
      install startYearSB using SpinBox::create $startLFfr.startYearSB \
      			-width 4 \
			-range {2000 2038 1} -editable yes -state disabled \
			-modifycmd [mymethod _updateDays start]
      $startYearSB setvalue first
      pack $startYearSB -side left
      pack [Label::create $startLFfr.s3 -text { }] -side left
      install startHrSB using SpinBox::create $startLFfr.startHrSB \
      			-width 2 \
			-range {0 23 1} -editable yes -state disabled
      $startHrSB setvalue first
      pack $startHrSB -side left
      pack [Label::create $startLFfr.colon -text {:}] -side left
      install startMinSB using SpinBox::create $startLFfr.startMinSB \
      			-width 2 \
			-range {0 59 1} -editable yes -state disabled
      $startMinSB setvalue first
      pack $startMinSB -side left
      # End time label frame. Contains the hour and minute of the end time.
      install endLF using LabelFrame::create $win.endLF \
				-text "End time:" \
				-width $::DatabaseWidgets::LabelWidth \
				-state disabled
      pack $endLF -fill x
      set endLFfr [$endLF getframe]
      install endMonthSB using SpinBox::create $endLFfr.endMonthSB \
			-width 2 \
			-range {1 12 1} -editable yes -state disabled \
			-modifycmd [mymethod _updateDays end]
      $endMonthSB setvalue first
      pack $endMonthSB -side left
      pack [Label::create $endLFfr.s1 -text {/}] -side left
      install endDaySB using SpinBox::create $endLFfr.endDaySB \
      			-width 2 \
			-range {1 31 1} -editable yes -state disabled
      $endDaySB setvalue first
      pack $endDaySB -side left
      pack [Label::create $endLFfr.s2 -text {/}] -side left
      install endYearSB using SpinBox::create $endLFfr.endYearSB \
      			-width 4 \
			-range {2000 2038 1} -editable yes -state disabled \
			-modifycmd [mymethod _updateDays end]
      $endYearSB setvalue first
      pack $endYearSB -side left
      pack [Label::create $endLFfr.s3 -text { }] -side left
      install endHrSB using SpinBox::create $endLFfr.endHrSB \
      			-width 2 \
			-range {0 23 1} -editable yes -state disabled
      $endHrSB setvalue first
      pack $endHrSB -side left
      pack [Label::create $endLFfr.colon -text {:}] -side left
      install endMinSB using SpinBox::create $endLFfr.endMinSB \
      			-width 2 \
			-range {0 59 1} -editable yes -state disabled
      $endMinSB setvalue first
      pack $endMinSB -side left
      # Distance traveled frame.  Contains the milage of the trip.
      install distanceLF using LabelFrame::create $win.distanceLF \
				-text "Distance:" \
				-width $::DatabaseWidgets::LabelWidth \
				-state disabled
      pack $distanceLF -fill x
      set distanceLFfr [$distanceLF getframe]
      install distMiSB using SpinBox::create $distanceLFfr.distMiSB \
			-width 5 \
			-range {0 999.9 .1} -editable yes -state disabled
      $distMiSB setvalue first
      pack $distMiSB -side left -fill x -expand yes -side left
      pack [Label::create $distanceLFfr.miles -text {Miles}] -side right
      # Maximum speed.  Contains the maximum speed achieved during the trip.
      install maxspeedLF using LabelFrame::create $win.maxspeedLF \
				-text "Maximum Speed:" \
				-width $::DatabaseWidgets::LabelWidth \
				-state disabled
      pack $maxspeedLF -fill x
      set maxspeedLFfr [$maxspeedLF getframe]
      install maxMPHSB using SpinBox::create $maxspeedLFfr.maxMPHSB \
			-width 4 \
			-range {0 99.9 .1} -editable yes -state disabled
      $maxMPHSB setvalue first
      pack $maxMPHSB -side left -fill x -expand yes -side left
      pack [Label::create $maxspeedLFfr.miles -text {Miles}] -side right
      # Commentary frame
      install commentsTF using TitleFrame::create $win.commentsTF \
		-text "Comments:" -side left -state disabled
      pack $commentsTF -expand yes -fill both
      set commentsTFfr [$commentsTF getframe]
      set commentsTFscroll [ScrolledWindow::create $commentsTFfr.scroll \
				-scrollbar both -auto both]
      pack $commentsTFscroll -expand yes -fill both
      install commentsText using text $commentsTFscroll.commentsText \
		-width 40 -height 8 -wrap word -state disabled
      pack $commentsText -expand yes -fill both
      $commentsTFscroll setwidget $commentsText
      # Add entry button
      install button using Button::create $win.button \
				-text "Add Trip" \
				-command [mymethod _AddTrip] \
				-state disabled
      pack $button -fill x
      #$self configurelist $args
      foreach {zmonth day year} [split [clock format [clock scan now] -format {%m %e %y}] { }] {break}
      scan "$zmonth" {%d} month
      $startMonthSB configure -text "$month"
      $endMonthSB configure -text "$month"
      $startDaySB configure -text "$day"
      $endDaySB configure -text "$day"
      $startYearSB  configure -text "$year"
      $endYearSB  configure -text "$year"
    }
    #***************************************
    # Method open: connect to a database. Create statement objects to 
    # add records and enable all GUI elements.
    #***************************************
    method open {connection} {
      catch {$self close};#		Close existing connection, if any
      set _Connection $connection;#	Save current connection
      # Create statement objects
      set _InsertTrip \
		[$_Connection \
			statement [DatabaseWidgets::statementID create] \
			"insert into $::DatabaseWidgets::TableName values (?,?,?,?,?,?,?)" \
			{TIMESTAMP TIMESTAMP REAL REAL INTEGER REAL CHAR}]
      # Enable all of the GUI elememnts
      $startLF configure -state normal
      $startMonthSB configure -state normal
      $startDaySB configure -state normal
      $startYearSB configure -state normal
      $startHrSB configure -state normal
      $startMinSB configure -state normal
      $endLF configure -state normal
      $endMonthSB configure -state normal
      $endDaySB configure -state normal
      $endYearSB configure -state normal
      $endHrSB configure -state normal
      $endMinSB configure -state normal
      $distanceLF configure -state normal
      $distMiSB configure -state normal
      $maxspeedLF configure -state normal
      $maxMPHSB configure -state normal
      $commentsTF configure -state normal
      $commentsText configure -state normal
      $button configure -state normal
    }
    #***************************************
    # Method close: drop statement object and disable all GUI elememnts.
    #***************************************
    method close {} {
      # Drop statement object
      catch {$_InsertTrip drop}
      set _InsertAddress {}
      # Reset connect
      set _Connection {}
      # Disable all GUI elememnts
      $startLF configure -state disabled
      $startMonthSB configure -state disabled
      $startDaySB configure -state disabled
      $startYearSB configure -state disabled
      $startHrSB configure -state disabled
      $startMinSB configure -state disabled
      $endLF configure -state disabled
      $endMonthSB configure -state disabled
      $endDaySB configure -state disabled
      $endYearSB configure -state disabled
      $endHrSB configure -state disabled
      $endMinSB configure -state disabled
      $distanceLF configure -state disabled
      $distMiSB configure -state disabled
      $maxspeedLF configure -state disabled
      $maxMPHSB configure -state disabled
      $commentsTF configure -state disabled
      $commentsText configure -state disabled
      $button configure -state disabled
    }
    #***************************************
    # Method _AddTrip: insert the contents of the GUI elements into the 
    # database.
    #***************************************
    method _AddTrip {} {
      # Make sure we have a table to add rows to.
      if {[catch {$_Connection tables $::DatabaseWidgets::TableName} tableinfo]} {
#	puts stderr "*** $self _AddTrip: tableinfo = $tableinfo"
        tk_messageBox -type ok -icon error -message "Please initialize the database first!"
	return
      }
      # Extract fields from the GUI and then insert the row.
      #-------
      set smin [format {%02d} [scan "[$startMinSB cget -text]" {%d}]]
      if {[catch {clock scan "[$startMonthSB cget -text]/[$startDaySB cget -text]/[$startYearSB cget -text] [$startHrSB cget -text]:$smin"} starttime]} {
	tk_messageBox -type ok -icon error -message "Please enter a proper start time!"
	return
      }
      set emin [format {%02d} [scan "[$endMinSB cget -text]" {%d}]]
      if {[catch {clock scan "[$endMonthSB cget -text]/[$endDaySB cget -text]/[$endYearSB cget -text] [$endHrSB cget -text]:$emin"} endtime]} {
	tk_messageBox -type ok -icon error -message "Please enter a proper end time!"
	return
      }
      set duration [expr {$endtime - $starttime}]
      if {$duration <= 0} {
	tk_messageBox -type ok -icon error -message "Please enter proper start and end times (end time is not later than start time)!"
	return
      }
      set distance [$distMiSB cget -text]
      set maxspeed [$maxMPHSB cget -text]
      set avespeed [expr {$distance / ($duration / 3600.0)}]
      set comments "[$commentsText get 1.0 end-1c]"
      # Insert into database
      set sdate "[clock format $starttime -format {%Y-%m-%d %H:%M:%S}]"
      set edate "[clock format $endtime -format {%Y-%m-%d %H:%M:%S}]"
#      puts stderr "*** $self _AddTrip: sdate = $sdate, edate = $edate"
      set res [catch {$_InsertTrip run [list \
			"$sdate" "$edate" \
			$distance $maxspeed $duration $avespeed \
			"$comments"]} error]
      if {$res} {tk_messageBox -type ok -icon error -message "$error"}
    }
    #***************************************
    # Destructor: close database connection, if any.
    #***************************************
    destructor {
      catch {$self close}
    }
    #***************************************
    # Method _updateDays -- update the range of days depending on the 
    # month and year
    #***************************************
    method _updateDays {which} {
      set monthSB [set ${which}MonthSB]
      set daySB [set ${which}DaySB]
      set yearSB [set ${which}YearSB]
      switch [$monthSB cget -text] {
	1 -
	3 -
	5 -
	7 -
	8 -
	10 -
	12 {
	    $daySB configure -range {1 31 1}
	}
	4 -
	6 -
	9 -
	11 {
	    $daySB configure -range {1 30 1}
	}
	2 {
	   $daySB configure -range {1 28 1}
	   set year [$yearSB cget -text]
	   set y4   [expr {int($year / 4)*4}]
 	   set y100 [expr {int($year / 100)*100}]
	   set y400 [expr {int($year / 400)*400}]
	   if {$y4 != $year} {return}
	   if {$y100 == $year} {if {$y400 == $year} {return}}
	   $daySB configure -range {1 29 1}
	}
      }
    }
  }
  #***************************************
  # General search frame
  #***************************************
  snit::widget SearchFrame {
    variable _Connection {};#		Database connection object
    variable _searchStatement {};#	Current search statement
    variable _Agregator AND;#		Agregrator
    variable _LastWhereClause {};#	Last Where clause

    # Search frame components
    component leftTF;#			Left (search terms) title frame
    component agregatorLF;#		Agregrator label frame
    component   andRB;#			And Radiobutton
    component   orRB;#			Or Radiobutton
    component starttimeLF
    component   starttimeCompCB
    component   starttimeMonthSB
    component   starttimeDaySB
    component   starttimeYearSB
    component   starttimeHourSB
    component   starttimeMinuteSB
    component endtimeLF
    component   endtimeCompCB
    component   endtimeMonthSB
    component   endtimeDaySB
    component   endtimeYearSB
    component   endtimeHourSB
    component   endtimeMinuteSB
    component distanceLF
    component   distanceCompCB
    component   distanceNumberSB
    component maxspeedLF
    component   maxspeedCompCB
    component   maxspeedNumberSB
    component durationLF
    component   durationCompCB
    component   durationHourSB
    component   durationMinuteSB
    component averagespeedLF
    component   averagespeedCompCB
    component   averagespeedNumberSB
    component searchButtons;#		Search buttons
    component rightTF;#			Right (where clause) title frame
    component clauseList;#		Where clause list
    component clauseButtons;#		Where clause buttons

    # Options:
    option -orderby;#			Order by field(s)
    option -selectfields -default {*};#	Selection fields
    option -rowfunction;#		Row function.
    option -searchresulthook;#		Search Results hook script
    option -searchstarthook;#		Search Start hook script
    option -callbackargs -default {};#	Callback args
    option -includesearch -default yes;# Include search button?
    #****************************************
    #* Helper method to create a date-time line item
    #****************************************
    method _searchElementLFDT {parent what label} {
      # Label frame
      install ${what}LF using LabelFrame::create $parent.${what}LF \
				-text "$label" \
				-width $::DatabaseWidgets::LabelWidth \
				-state disabled
      pack [set ${what}LF] -fill x
      set ${what}fr [[set ${what}LF] getframe]
      # Comparitor combo box
      install ${what}CompCB using ComboBox::create [set ${what}fr].${what}CompCB \
				-editable no \
				-values {<= >= <> =} -width 2 \
				-state disabled
      [set ${what}CompCB] setvalue first
      pack [set ${what}CompCB] -side left
      # Compare date/time entry
      install ${what}MonthSB using SpinBox::create \
		[set ${what}fr].${what}MonthSB \
      			-width 2 -state disabled\
			-range {1 12 1} -editable yes \
			-modifycmd [mymethod _updateDays $what]
      [set ${what}MonthSB] configure -text {-}
      pack [set ${what}MonthSB] -side left
      pack [Label::create [set ${what}fr].s1 -text {/}] -side left
      install ${what}DaySB using SpinBox::create \
		[set ${what}fr].${what}DaySB \
			-width 2 \
			-range {1 31 1} -editable yes -state disabled
      [set ${what}DaySB] configure -text {-}
      pack [set ${what}DaySB] -side left
      pack [Label::create [set ${what}fr].s2 -text {/}] -side left
      install ${what}YearSB  using SpinBox::create \
		[set ${what}fr].${what}YearSB \
			-width 4 \
			-range {2000 2038 1} -editable yes -state disabled \
			-modifycmd [mymethod _updateDays $what]
      [set ${what}YearSB] configure -text {-}
      pack [set ${what}YearSB] -side left
      pack [Label::create [set ${what}fr].s3 -text { }] -side left
      install ${what}HourSB using SpinBox::create \
		[set ${what}fr].${what}HourSB \
			-width 2 \
			-range {0 23 1} -editable yes -state disabled
      [set ${what}HourSB] configure -text {-}
      pack [set ${what}HourSB] -side left
      pack [Label::create [set ${what}fr].s4 -text {:}] -side left
      install ${what}MinuteSB using SpinBox::create \
		[set ${what}fr].${what}MinuteSB \
			-width 2 \
			-range {0 59 1} -editable yes -state disabled
      [set ${what}MinuteSB] configure -text {-}
      pack [set ${what}MinuteSB] -side left
    }
    #****************************************
    #* Helper method to create a number SpinBox line item
    #****************************************
    method _searchElementLFNSB {parent what label range} {
      # Label frame
      install ${what}LF using LabelFrame::create $parent.${what}LF \
				-text "$label" \
				-width $::DatabaseWidgets::LabelWidth \
				-state disabled
      pack [set ${what}LF] -fill x
      set ${what}fr [[set ${what}LF] getframe]
      # Comparitor combo box
      install ${what}CompCB using ComboBox::create [set ${what}fr].${what}CompCB \
				-editable no \
				-values {< <= = <> >= >} -width 2 \
				-state disabled
      [set ${what}CompCB] setvalue first
      pack [set ${what}CompCB] -side left
      # Compare string entry
      install ${what}NumberSB using SpinBox::create \
			[set ${what}fr].${what}NumberSB \
					-state disabled \
					-editable yes -range $range
      [set ${what}NumberSB] configure -text {-}
      pack [set ${what}NumberSB] -side left -fill x -expand yes
    }
        #***************************************
    # Constructor: populate GUI elements
    #***************************************
    constructor {args} {
      $self configurelist $args;#		Get our options
      # Left side title frame: clause building GUI elements
      install leftTF using TitleFrame::create $win.leftTF \
				-text "Search Terms" \
				-side center -state disabled
      pack $leftTF -side left -expand yes -fill both
      set left [$leftTF getframe]
      # Clause agregator RadioButtons
      install agregatorLF using LabelFrame::create $left.agregatorLF \
				-text "Agregator:" \
				-width $::DatabaseWidgets::LabelWidth \
				-state disabled
      pack $agregatorLF -fill x
      set agregatorfr [$agregatorLF getframe]
      install andRB using radiobutton $agregatorfr.andRB -text AND -value AND \
				      -variable [myvar _Agregator] \
				      -state disabled
      pack $andRB -fill x -side left -expand yes
      install orRB using radiobutton $agregatorfr.orRB -text OR -value OR \
				      -variable [myvar _Agregator] \
				      -state disabled
      pack $orRB -fill x -side left -expand yes
      # Search elements (clause items)
      $self _searchElementLFDT $left starttime {Start Time:}
      $self _searchElementLFDT $left endtime {End Time:}
      $self _searchElementLFNSB $left distance {Distance:} {0 999.9 .1}
      $self _searchElementLFNSB $left maxspeed {Maximum Speed:} {0 99.9 .1}
      # Duration is a time (HH:MM)
      # Label frame
      install durationLF using LabelFrame::create $left.durationLF \
				-text {Duration:} \
				-width $::DatabaseWidgets::LabelWidth \
				-state disabled
      pack $durationLF -fill x
      set durationLFfr [$durationLF  getframe]
      # Comparitor combo box
      install durationCompCB using \
		ComboBox::create $durationLFfr.durationCompCB \
			-editable no \
			-values {< <= = <> >= >} -width 2 \
			-state disabled
      $durationCompCB setvalue first
      pack $durationCompCB -side left
      # Compare time entry
      install durationHourSB using SpinBox::create \
		$durationLFfr.durationHourSB -width 2 -range {0 23 1} \
			-editable yes -state disabled
      $durationHourSB configure -text {-}
      pack $durationHourSB -side left
      pack [Label::create $durationLFfr.s1 -text {:}] -side left
      install durationMinuteSB using SpinBox::create \
		$durationLFfr.durationMinuteSB -width 2 -range {0 59 1} \
			-editable yes -state disabled
      $durationMinuteSB configure -text {-}
      pack $durationMinuteSB -side left
      $self _searchElementLFNSB $left averagespeed {Average Speed:} {0 99.9 .1}
      # Search buttons (left side)
      install searchButtons using ButtonBox::create $left.searchButtons \
				-orient horizontal -homogeneous yes\
				-state disabled
      pack $searchButtons -fill x -side bottom
      # Dialog boxes have their Search button at the bottom
      if {$options(-includesearch)} {
        $searchButtons add -name search -text "Search" \
				-command [mymethod dosearch] \
				-state disabled
      }
      # Just add the clause
      $searchButtons add -name addclause -text "Add Clause" \
				-command [mymethod _AddClause] \
				-state disabled
      install rightTF using TitleFrame::create $win.rightTF \
				-text "Where clause" \
				-side center -state disabled
      # Right side title frame: list of clauses
      pack $rightTF -side right -expand yes -fill both
      set right [$rightTF getframe]
      set clauseListSW [ScrolledWindow::create $right.clauseListSW \
						-scrollbar both -auto both]
      pack $clauseListSW -expand yes -fill both
      install clauseList using ListBox::create \
				[$clauseListSW getframe].clauseList \
				-selectfill yes -selectmode single \
				-state disabled
      pack $clauseList -expand yes -fill both
      $clauseListSW setwidget $clauseList
      # Clause buttons: delete one or all clauses
      install clauseButtons using ButtonBox::create $right.clauseButtons \
				-orient horizontal -homogeneous yes\
				-state disabled
      pack $clauseButtons -fill x -side bottom
      $clauseButtons add -name deleteClause -text "Delete Clause" \
      				-command [mymethod _DeleteClause] \
				-state disabled
      $clauseButtons add -name newSearch -text "New Search" \
      				-command [mymethod _NewSearch] \
				-state disabled
    }
    #***************************************
    # open method -- open things up.  Get connected to the database and
    # enable all of the GUI elements
    #***************************************
    method open {connection} {
      catch {$self close};#             Close existing connection, if any
      set _Connection $connection
      $leftTF configure -state normal
      $agregatorLF configure -state normal
      $andRB configure -state normal
      $orRB  configure -state normal
      $starttimeLF  configure -state normal
      $starttimeCompCB  configure -state normal
      $starttimeMonthSB  configure -state normal
      $starttimeDaySB  configure -state normal
      $starttimeYearSB  configure -state normal
      $starttimeHourSB  configure -state normal
      $starttimeMinuteSB  configure -state normal
      $endtimeLF  configure -state normal
      $endtimeCompCB  configure -state normal
      $endtimeMonthSB  configure -state normal
      $endtimeDaySB  configure -state normal
      $endtimeYearSB  configure -state normal
      $endtimeHourSB  configure -state normal
      $endtimeMinuteSB  configure -state normal
      $distanceLF  configure -state normal
      $distanceCompCB  configure -state normal
      $distanceNumberSB  configure -state normal
      $maxspeedLF  configure -state normal
      $maxspeedCompCB  configure -state normal
      $maxspeedNumberSB  configure -state normal
      $durationLF  configure -state normal
      $durationCompCB  configure -state normal
      $durationHourSB  configure -state normal
      $durationMinuteSB  configure -state normal
      $averagespeedLF  configure -state normal
      $averagespeedCompCB  configure -state normal
      $averagespeedNumberSB  configure -state normal
      $searchButtons configure -state normal
      $rightTF configure -state normal
      $clauseList configure -state normal
      $clauseButtons configure -state normal
    }
    #***************************************
    # close method -- disconnect from the database and disable all of the GUI 
    # elements
    #***************************************
    method close {} {
      set _Connection {}
      catch {$_searchStatement drop}
      set _searchStatement {}
      $clauseList delete [$clauseList items]
      $leftTF configure -state disabled
      $agregatorLF configure -state disabled
      $andRB configure -state disabled
      $orRB  configure -state disabled
      $starttimeLF configure -state disabled
      $starttimeCompCB configure -state disabled
      $starttimeMonthSB configure -state disabled
      $starttimeDaySB configure -state disabled
      $starttimeYearSB configure -state disabled
      $starttimeHourSB configure -state disabled
      $starttimeMinuteSB configure -state disabled
      $endtimeLF configure -state disabled
      $endtimeCompCB configure -state disabled
      $endtimeMonthSB configure -state disabled
      $endtimeDaySB configure -state disabled
      $endtimeYearSB configure -state disabled
      $endtimeHourSB configure -state disabled
      $endtimeMinuteSB configure -state disabled
      $distanceLF configure -state disabled
      $distanceCompCB configure -state disabled
      $distanceNumberSB configure -state disabled
      $maxspeedLF configure -state disabled
      $maxspeedCompCB configure -state disabled
      $maxspeedNumberSB configure -state disabled
      $durationLF configure -state disabled
      $durationCompCB configure -state disabled
      $durationHourSB configure -state disabled
      $durationMinuteSB configure -state disabled
      $averagespeedLF configure -state disabled
      $averagespeedCompCB configure -state disabled
      $averagespeedNumberSB configure -state disabled
      $searchButtons configure -state disabled
      $rightTF configure -state disabled
      $clauseList configure -state disabled
      $clauseButtons configure -state disabled
    }
    #***************************************
    # destructor -- make sure the connection is closed.
    #***************************************
    destructor {
      catch {$self close}
    }
    method whereclause {} {return "$_LastWhereClause"}
    #***************************************
    # Method _updateDays -- update the range of days depending on the 
    # month and year
    #***************************************
    method _updateDays {which} {
      set monthSB [set ${which}MonthSB]
      set daySB [set ${which}DaySB]
      set yearSB [set ${which}YearSB]
      switch [$monthSB cget -text] {
	1 -
	3 -
	5 -
	7 -
	8 -
	10 -
	12 {
	    $daySB configure -range {1 31 1}
	}
	4 -
	6 -
	9 -
	11 {
	    $daySB configure -range {1 30 1}
	}
	2 {
	   $daySB configure -range {1 28 1}
	   set year [$yearSB cget -text]
	   set y4   [expr {int($year / 4)*4}]
 	   set y100 [expr {int($year / 100)*100}]
	   set y400 [expr {int($year / 400)*400}]
	   if {$y4 != $year} {return}
	   if {$y100 == $year} {if {$y400 == $year} {return}}
	   $daySB configure -range {1 29 1}
	}
      }
    }
    #***************************************
    # _CheckFieldNSB method -- check to see if the selected field has a non 
    # null compare string and if so, return the clause
    #***************************************
    method _CheckFieldNSB {name} {
      set val "[[set ${name}NumberSB] cget -text]"
      if {[string equal "$val" {-}]} {
	return {}
      } else {
        [set ${name}NumberSB] configure -text {-}
	return [list "$name [[set ${name}CompCB] cget -text] ?" $val]
      }
    }
    #***************************************
    # _CheckFieldDT method -- check to see if the selected field has a non 
    # null compare string and if so, return the clause
    #***************************************
    method _CheckFieldDT {name} {
      set month [[set ${name}MonthSB] cget -text]
      set day [[set ${name}DaySB] cget -text]
      set year [[set ${name}YearSB] cget -text]
      set hour [[set ${name}HourSB] cget -text]
      set minute [[set ${name}MinuteSB] cget -text]
      if {[string equal "$month" {-}] ||
	  [string equal "$day" {-}] ||
	  [string equal "$year" {-}] ||
	  [string equal "$hour" {-}] ||
	  [string equal "$minute" {-}]} {
	return {}
      } else {
	[set ${name}MonthSB] configure -text {-}
	[set ${name}DaySB] configure -text {-}
	[set ${name}YearSB] configure -text {-}
	[set ${name}HourSB] configure -text {-}
	[set ${name}MinuteSB] configure -text {-}
	set minute [format {%02d} [scan "$minute" {%d}]]
	set timeSecs [clock scan "$month/$day/$year $hour:$minute"]
	return [list "$name [[set ${name}CompCB] cget -text] ?" \
			"[clock format $timeSecs -format {%Y-%m-%d %H:%M:%S}]"]
      }
    }
    #***************************************
    # _AddClause method -- check all fields and add clauses.
    #***************************************
    method _AddClause {} {
      foreach n {starttime endtime} {
	if {[llength [$clauseList items]] > 0} {
          set newclause "$_Agregator "
	} else {
	  set newclause {}
	}
	set cL [$self _CheckFieldDT $n]
	if {[llength $cL] < 2} {continue}
	append newclause "[lindex $cL 0]"
	$clauseList insert end #auto \
			-text [regsub {\?} "$newclause" "'[lindex $cL 1]'"] \
			-data [list $newclause "[lindex $cL 1]" TIMESTAMP]
      }
      foreach n {distance maxspeed averagespeed} {
	if {[llength [$clauseList items]] > 0} {
          set newclause "$_Agregator "
	} else {
	  set newclause {}
	}
	set cL [$self _CheckFieldNSB $n]
	if {[llength $cL] < 2} {continue}
	append newclause "[lindex $cL 0]"
	$clauseList insert end #auto \
			-text [regsub {\?} "$newclause" "[lindex $cL 1]"] \
			-data [list $newclause "[lindex $cL 1]" REAL]
      }
      set durationH [$durationHourSB cget -text]
      set durationM [$durationMinuteSB cget -text]
      set durationC [$durationCompCB cget -text]
      if {![string equal "$durationH" {-}] &&
	  ![string equal "$durationM" {-}]} {
	set duration [expr {($durationH * 3600) + ($durationM * 60)}]
	$durationHourSB configure -text {-}
	$durationMinuteSB configure -text {-}
	set cL [list "duration $durationC ?" $duration]
	append newclause "[lindex $cL 0]"
        set durfmt "[clock format $duration -format {%H:%M} -gmt yes]"
	$clauseList insert end #auto \
			-text [regsub {\?} "$newclause" "'$durfmt'"] \
			-data [list $newclause "[lindex $cL 1]" INTEGER]
      }
    }
    #***************************************
    # dosearch method -- perform a search.
    #***************************************
    method dosearch {} {
      # If the search buttons are disabled, abort the search -- there
      # is no database connection
      set state [$searchButtons cget -state]
      if {![string equal "$state" normal]} {return}
      # Add any clauses
      $self _AddClause
      # Get clause list
      set clauses [$clauseList items]
      # Start to build SQL string
      set sql "select $options(-selectfields) from $::DatabaseWidgets::TableName"
      set arglist {}
      set argdefs {}
      set _LastWhereClause {}
      # Loop over clauses, adding clauses, arglist elements, and argdef elements
      if {[llength $clauses] > 0} {
	append sql " where"
	foreach cl $clauses {
	  set cL [$clauseList itemcget $cl -data]
	  append sql " [lindex $cL 0]"
	  append _LastWhereClause [$clauseList itemcget $cl -text]
	  lappend arglist [lindex $cL 1]
	  lappend argdefs [lindex $cL 2]
	}
      }
#      # Add order clause
      set orderby "$options(-orderby)"
      if {[string length "$orderby"] > 0} {
	append sql " order by $orderby"
      }
#      puts stderr "*** $self dosearch: sql = $sql"
      # Drop an old statement
      catch {$_searchStatement drop}
      # Create statement object
      set _searchStatement [$_Connection statement \
				[DatabaseWidgets::statementID create] "$sql" $argdefs]
      # Call start hook, if any
      if {[string length "$options(-searchstarthook)"] > 0} {
	set result [uplevel #0 $options(-searchstarthook) $options(-callbackargs)]
#	puts stderr "*** $self dosearch: (-searchstarthook): result = $result"
	if {!$result} {return}
      }
      # Start search
      $_searchStatement execute $arglist
      # Loop over rows
      set rowcount 0
      while {[set row [$_searchStatement fetch]] != {}} {
        # Call row function
	uplevel #0 $options(-rowfunction) [list $row] $options(-callbackargs)
#	puts stderr "*** $self dosearch: (-rowfunction): row = $row"
	incr rowcount
      }
#      puts stderr "*** $self dosearch: rowcount = $rowcount"
      # Call result hook
      if {[string length "$options(-searchresulthook)"] > 0} {
	uplevel #0 $options(-searchresulthook) $rowcount $options(-callbackargs)
      }
    }
    #***************************************
    # _DeleteClause method -- delete selected clause
    #***************************************
    method _DeleteClause {} {
      # Get selected clauses
      set selection [$clauseList selection get]
      # If none selected, return
      if {[llength $selection] == 0} {return}
      # Delete selected clause
      set selection [lindex $selection 0]
      $clauseList delete $selection
      set remainingItems [$clauseList items]
      if {[llength $remainingItems] == 0} {return}
      # Check first clause and remove agregator if any
      set first [lindex $remainingItems 0]
      set toptext "[$clauseList itemcget $first -text]"
      set topCl    [$clauseList itemcget $first -data]
      if {[regexp {^(AND|OR) } "$toptext"] > 0} {
	regsub {^(AND|OR) } "$toptext" {} toptext
	set clause [lindex $topCl 0]
	regsub {^(AND|OR) } "$clause" {} clause
	set topCl [list $clause [lindex $topCl 1] [lindex $topCl 2]]
	$clauseList itemconfigure $first -text "$toptext" -data $topCl
      }
    }
    #***************************************
    # _NewSearch method -- delete all clauses.
    #***************************************
    method _NewSearch {} {
      $clauseList delete [$clauseList items]
    }
  }
  #***************************************
  # Macro to create a base Search frame in a dialog
  #***************************************
  snit::macro DatabaseWidgets::SearchDialogConstructor {helptopic btext body} {
    # Common options, both delegated to the hull (a Dialog)
    delegate option -title to hull;#	Title
    delegate option -parent to hull;#	Parent
    component searchFR;#		Search frame
    # Stock constructor body: Build the standard dialog with a search frame.
    # Other GUI elements are added below the search frame.
    set cbody {
	installhull using Dialog::create -modal local \
			-bitmap question -default 0 -cancel 1 -side bottom \
			-parent .
        $self configurelist $args
        $hull add -name search -text %%BTEXT%% -command [mymethod _Search]
	$hull add -name cancel -text Cancel -command [mymethod _Cancel]
	$hull add -name help -text Help \
			     -command "BWHelp::HelpTopic %%HELPTOPIC%%"
	set frame [$hull getframe]
	install searchFR using DatabaseWidgets::SearchFrame \
				$frame.searchFR \
				-rowfunction [mymethod _RowFunction]\
				-searchresulthook [mymethod _ResultHook]\
				-searchstarthook [mymethod _StartHook]\
				-includesearch no
	pack $searchFR -expand yes -fill both
    }
    # Substitute in custom values:
    regsub -all {%%HELPTOPIC%%} "$cbody" "$helptopic" cbody
    regsub -all {%%BTEXT%%} "$cbody" "$btext" cbody
    # Append the rest of the body
    append cbody "$body"
    # Build constructor
    constructor {args} "$cbody"
    # Delegate the open and close methods to the search frame
    delegate method open to searchFR
    delegate method close to searchFR
    # Search button binding
    method _Search {} {
      $searchFR dosearch
      $hull withdraw
      return [$hull enddialog search]
    }
    # Cancel button binding
    method _Cancel {} {
      $hull withdraw
      return [$hull enddialog cancel]
    }
    # Draw method
    method draw {args} {
      $self configurelist $args
      wm transient [winfo toplevel $win] [$hull cget -parent]
      $self _DrawHook;# Hook function for drawing the dialog
      return [$hull draw]
    }
  }
  #**********************************
  # Bar chart frame
  #**********************************
  snit::widget BarChartFrame {

    delegate option -chartheight to chartCanvas as -height
    delegate option -chartwidth to chartCanvas as -width

    # Frame components
    component chartSW;#			The chart canvas scrollbars
    component   chartCanvas;#		  The chart canvas
    component flagsLF;#			Display flags LF
    component   durationCheckbut;#	  Duration flag
    component   averageCheckbut;#	  Average speed flag
    component   maximumCheckbut;#	  Maximum speed flag
    component   distanceCheckbut;#	  Distance flag
    component searchFrame;#		Search Frame
    component button;#                  Draw Chart Button

    variable _ShowDuration yes;#	Show duration bar		
    variable _ShowAverage  yes;#	Show average speed bar
    variable _ShowMaximum  no;#		Show maximum speed bar
    variable _ShowDistance yes;#	Show distance bar
    variable _HaveChart    no;#		Have a chart ready to print?
    #***************************************
    # Constructor: populate frame with GUI elememnts.
    #***************************************
    constructor {args} {
      # Create the drawing area (scrolled canvas)
      install chartSW using ScrolledWindow::create $win.chartSW \
				-scrollbar both -auto both
      pack $chartSW -expand yes -fill both
      install chartCanvas using canvas $chartSW.chartCanvas \
				-background white -relief sunken
      pack $chartCanvas -expand yes -fill both
      $chartSW setwidget $chartCanvas
      # Display flag frame
      install flagsLF using LabelFrame::create $win.flagsLF \
			-text "Show:" \
			-width $::DatabaseWidgets::LabelWidth \
			-state disabled
      pack $flagsLF -fill x
      set flagsLFfr [$flagsLF getframe]
      foreach name {durationCheckbut averageCheckbut maximumCheckbut distanceCheckbut} \
	      text {Duration {Ave. Speed} {Max. Speed} Distance} \
	      var  {_ShowDuration _ShowAverage _ShowMaximum _ShowDistance} {
	install $name using checkbutton $flagsLFfr.$name \
		-indicatoron yes -offvalue no -onvalue yes \
		-state disabled -variable [myvar $var] \
		-text "$text"
	pack [set $name] -side left
      }
      # Search frame
      install searchFrame using ::DatabaseWidgets::SearchFrame \
					$win.searchFrame -includesearch no
      pack $searchFrame -expand yes -fill both
      # Draw chart button
      install button using Button::create $win.button \
			-text "Draw Chart" \
			-command [mymethod _DrawChart] \
			-state disabled
      pack $button -fill x
      $self configurelist $args
    }
    #***************************************
    # Method open: connect to a database. Create statement objects to 
    # add records and enable all GUI elements.
    #***************************************
    method open {connection} {
      catch {$self close};#		Close existing connection, if any
      $searchFrame open $connection
      # Enable all of the GUI elememnts
      $flagsLF configure -state normal
      $durationCheckbut configure -state normal
      $averageCheckbut configure -state normal
      $maximumCheckbut configure -state normal
      $distanceCheckbut configure -state normal
      $button configure -state normal
    }
    #***************************************
    # Method close: drop statement object and disable all GUI elememnts.
    #***************************************
    method close {} {
      $searchFrame close
      # Disable all GUI elememnts
      $flagsLF configure -state disabled
      $durationCheckbut configure -state disabled
      $averageCheckbut configure -state disabled
      $maximumCheckbut configure -state disabled
      $distanceCheckbut configure -state disabled
      $button configure -state disabled
    }
    variable barcount 0
    variable maxduration 0
    variable maxavespeed 0
    variable maxmaxspeed 0
    variable maxdistance 0

    method _GetRanges {row} {
      set barcount     [lindex $row 0]
      set maxduration  [lindex $row 1]
      set maxavespeed  [lindex $row 2]
      set maxmaxspeed  [lindex $row 3]
      set maxdistance  [lindex $row 4]
    }
    typevariable _ColorList {red green blue orange}
    typevariable _BarHeight     280.0
    variable barindex 0
    variable xOff
    variable bOff
    variable bWidth
    variable bYZero
    variable b1Off
    variable b1Width
    variable durationscale
    variable speedscale
    variable distancescale
    method _DrawBars {row} {
      set colors $_ColorList
      foreach {starttime distance maxspeed duration avespeed} $row {break}
        set bCenter [expr {$xOff + $bOff + ($barindex * $bWidth)}]
      $chartCanvas create text $bCenter  [expr {$bYZero + 10}] \
            -anchor n \
            -text  [clock format [clock scan $starttime] -format "%b %d, %Y\n%H:%M"]
      set xB1 [expr {$bCenter - $bOff + $b1Off}]
      if {$_ShowDuration} {
	set color [lindex $colors 0]
	set colors [lrange $colors 1 end]
	set bHeight [expr {$duration * $durationscale}]
	set yTop [expr {$bYZero - $bHeight}]
	$chartCanvas create rect $xB1 $bYZero [expr {$xB1 + $b1Width}] $yTop \
	    -fill $color -outline {}
	set xB1 [expr {$xB1 + $b1Width}]
      }
      if {$_ShowAverage} {
	set color [lindex $colors 0]
	set colors [lrange $colors 1 end]
	set bHeight [expr {$avespeed * $speedscale}]
	set yTop [expr {$bYZero - $bHeight}]
	$chartCanvas create rect $xB1 $bYZero [expr {$xB1 + $b1Width}] $yTop \
	    -fill $color -outline {}
	set xB1 [expr {$xB1 + $b1Width}]
      }
      if {$_ShowMaximum} {
	set color [lindex $colors 0]
	set colors [lrange $colors 1 end]
	set bHeight [expr {$maxspeed * $speedscale}]
	set yTop [expr {$bYZero - $bHeight}]
	$chartCanvas create rect $xB1 $bYZero [expr {$xB1 + $b1Width}] $yTop \
	    -fill $color -outline {}
	set xB1 [expr {$xB1 + $b1Width}]
      }
      if {$_ShowDistance} {
	set color [lindex $colors 0]
	set colors [lrange $colors 1 end]
	set bHeight [expr {$distance * $distancescale}]
	set yTop [expr {$bYZero - $bHeight}]
	$chartCanvas create rect $xB1 $bYZero [expr {$xB1 + $b1Width}] $yTop \
	    -fill $color -outline {}
	set xB1 [expr {$xB1 + $b1Width}]
      }
      incr barindex
    }
    #***************************************
    # Method _DrawChart -- Draw the chart
    #***************************************
    method _DrawChart {} {
      set bars 0
      if {$_ShowMaximum} {incr bars}
      if {$_ShowAverage} {incr bars}
      if {$_ShowDuration} {incr bars}
      if {$_ShowDistance} {incr bars}
      if {$bars == 0} {
	tk_messageBox -type ok -icon warning -message "Please select at least one bar!"
      return
      }
      # First pass: get count, mins, and maxes.
      set selection "COUNT(*), MAX(duration), MAX(averagespeed), MAX(maxspeed), MAX(distance)"
      $searchFrame configure \
	-orderby {} \
	-selectfields "$selection" \
	-rowfunction [mymethod _GetRanges] \
	-searchresulthook {} \
	-searchstarthook {} \
	-callbackargs {}
      $searchFrame dosearch
      set maxdistance [$type _RoundUp $maxdistance 5]
      set maxduration [$type _RoundUp $maxduration 600]
      if {$barcount == 0} {
	tk_messageBox -type ok -icon warning -message "No data selected!"
      return
      }
#      puts stderr "*** $self _DrawChart: barcount = $barcount, maxduration = $maxduration, maxavespeed = $maxavespeed, maxmaxspeed = $maxmaxspeed, maxdistance = $maxdistance"
      set tempId [$chartCanvas create text 0 0 -anchor nw -text [clock format 0 -format "%b %d, %Y\n%H:%M"]]
      set dtbbox [$chartCanvas bbox $tempId]
      set bWidth [expr {[lindex $dtbbox 2] + 30}]
      set b1Width [expr {$bWidth / ($bars + 2)}]
      set b1Off   $b1Width
      set bOff   [expr {double($bWidth) / 2.0}]
      set bYZero  [expr {$_BarHeight + 10}]
      set bLHeight [lindex $dtbbox 3]
      set durationscale [expr {$_BarHeight / double($maxduration)}]
      set distancescale [expr {$_BarHeight / double($maxdistance)}]
      if {$_ShowMaximum} {
	set maxspeed $maxmaxspeed
      } else {
	set maxspeed $maxavespeed
      }
      set maxspeed [$type _RoundUp $maxspeed 1]
      set speedscale  [expr {$_BarHeight / double($maxspeed)}]
      set tempId [$chartCanvas create text 0 0 -anchor nw -text "MPH"]
      set spwidth [lindex [$chartCanvas bbox $tempId] 2]
      set tempId [$chartCanvas create text 0 0 -anchor nw -text "[format {%5.2f} 0.0]"]
      set distwidth [lindex [$chartCanvas bbox $tempId] 2]
      set tempId [$chartCanvas create text 0 0 -anchor nw -text "[clock format 0 -format {%H:%M} -gmt yes]"]
      set durwidth [lindex [$chartCanvas bbox $tempId] 2]
      $chartCanvas delete all
      set xOff 5
      set  x [expr {$xOff + $durwidth}]
      if {$_ShowDuration} {
	foreach sp {0 .1 .2 .3 .4 .5 .6 .7 .8 .9 1.0} {
	  set tick [expr {$sp * $_BarHeight}]
	  set duration [expr {int($sp * $maxduration)}]
	  set y [expr {$bYZero - $tick}]
	  $chartCanvas create text $x $y -anchor e -text "[clock format $duration -format {%H:%M} -gmt yes]"
	}
	set xOff [expr {$xOff + $durwidth + 5}]
	$chartCanvas create text $x [expr {$bYZero + 10}] -text "Time" -anchor ne
      }
      set x [expr {$xOff + $spwidth}]
      if {$_ShowMaximum || $_ShowAverage} {
	foreach sp {0 .1 .2 .3 .4 .5 .6 .7 .8 .9 1.0} {
	  set tick [expr {$sp * $_BarHeight}]
	  set speed [expr {$sp * $maxspeed}]
	  set y [expr {$bYZero - $tick}]
	  $chartCanvas create text $x $y -anchor e -text "[format {%4.1f} $speed]"
	}
	$chartCanvas create text $x [expr {$bYZero + 10}] -text "MPH" -anchor ne
	set xOff [expr {$xOff + $spwidth + 5}]
      }
      set  x [expr {$xOff + $distwidth}]
      if {$_ShowDistance} {
	foreach sp {0 .1 .2 .3 .4 .5 .6 .7 .8 .9 1.0} {
	  set tick [expr {$sp * $_BarHeight}]
	  set distance [expr {$sp * $maxdistance}]
	  set y [expr {$bYZero - $tick}]
	  $chartCanvas create text $x $y -anchor e -text "[format {%5.2f} $distance]"
	}
	$chartCanvas create text $x [expr {$bYZero + 10}] -text "Miles" -anchor ne
	set xOff [expr {$xOff + $distwidth + 5}]
      }
      set xt [expr {$xOff + 10}]
      set totBWidth [expr {$barcount * $bWidth}]
      foreach sp {0 .1 .2 .3 .4 .5 .6 .7 .8 .9 1.0} {
	set tick [expr {$sp * $_BarHeight}]
	set y [expr {$bYZero - $tick}]
	$chartCanvas create line $xOff $y $xt $y
	if {$sp > 0} {
	  $chartCanvas create line $xt $y [expr {$xt + $totBWidth}] $y -dash -
	} else {
	  $chartCanvas create line $xt $y [expr {$xt + $totBWidth}] $y
	}
      }
      set xOff $xt
      $chartCanvas create line $xOff $bYZero $xOff [expr {$bYZero - $_BarHeight}]
      
      set lX [expr {$xOff + 10}]
      set lY [expr {$bYZero + 10 + $bLHeight}]
      set colors $_ColorList
      if {$_ShowDuration} {
	set color [lindex $colors 0]
	set colors [lrange $colors 1 end]
	set id [$chartCanvas create text $lX $lY -text "Duration " -anchor nw]
	set bbox [$chartCanvas bbox $id]
	set bx1 [lindex $bbox 2]
	set by1 $lY
	set by2 [lindex $bbox 3]
	set bx2 [expr {$bx1 + ($by2 - $by1)}]
	$chartCanvas create rect $bx1 $by1 $bx2 $by2 -fill $color -outline grey
	set lY [expr {$by2 + 5}]
      }
      if {$_ShowAverage} {
	set color [lindex $colors 0]
	set colors [lrange $colors 1 end]
	set id [$chartCanvas create text $lX $lY -text "Ave. Speed " -anchor nw]
	set bbox [$chartCanvas bbox $id]
	set bx1 [lindex $bbox 2]
	set by1 $lY
	set by2 [lindex $bbox 3]
	set bx2 [expr {$bx1 + ($by2 - $by1)}]
	$chartCanvas create rect $bx1 $by1 $bx2 $by2 -fill $color -outline grey
	set lY [expr {$by2 + 5}]
      }
      if {$_ShowMaximum} {
	set color [lindex $colors 0]
	set colors [lrange $colors 1 end]
	set id [$chartCanvas create text $lX $lY -text "Max. Speed " -anchor nw]
	set bbox [$chartCanvas bbox $id]
	set bx1 [lindex $bbox 2]
	set by1 $lY
	set by2 [lindex $bbox 3]
	set bx2 [expr {$bx1 + ($by2 - $by1)}]
	$chartCanvas create rect $bx1 $by1 $bx2 $by2 -fill $color -outline grey
	set lY [expr {$by2 + 5}]
      }
      if {$_ShowDistance} {
	set color [lindex $colors 0]
	set colors [lrange $colors 1 end]
	set id [$chartCanvas create text $lX $lY -text "Distance " -anchor nw]
	set bbox [$chartCanvas bbox $id]
	set bx1 [lindex $bbox 2]
	set by1 $lY
	set by2 [lindex $bbox 3]
	set bx2 [expr {$bx1 + ($by2 - $by1)}]
	$chartCanvas create rect $bx1 $by1 $bx2 $by2 -fill $color -outline grey
	set lY [expr {$by2 + 5}]
      }
      # Second pass: draw the chart
      set barindex 0
      set selection "starttime, distance, maxspeed, duration, averagespeed"
      $searchFrame configure \
        -orderby {starttime} \
	-selectfields "$selection" \
	-rowfunction [mymethod _DrawBars] \
	-searchresulthook {} \
	-searchstarthook {} \
	-callbackargs {}
      $searchFrame dosearch
      $chartCanvas configure -scrollregion [$chartCanvas bbox all]
      set _HaveChart yes
    }
    method printP {} {
      return yes
    }
    method print {} {
      if {!$_HaveChart} {return}
#      if {[catch {open "|lpr" w} lpfp]} {return}
#      if {[catch {open "test.ps" w} lpfp]} {return}
      set pageheight 10i
      set pagewidth   8i
      set landscape  yes
      set lpfp [Printer::OpenPrinter "Print Chart" $win pageheight pagewidth landscape]
      if {[string equal "$lpfp" {}]} {return}
      set sr [$chartCanvas cget -scrollregion]
      set height [expr {[lindex $sr 3] - [lindex $sr 1] + 1}]
      set width  [expr {[lindex $sr 2] - [lindex $sr 0] + 1}]
      set y      [lindex $sr 1]
      set x      [lindex $sr 0]
#      puts stderr "*** $self print: sr = $sr, height = $height, width = $width, y = $y x = $x"
      $chartCanvas postscript -channel $lpfp -height $height -width $width \
      			      -x $x -y $y -rotate $landscape \
			      -pageheight $pageheight -pagewidth $pagewidth
      close $lpfp
    }
    typemethod _RoundUp {value multiple} {
      set q [expr {int(ceil(double($value) / double($multiple)))}]
      return [expr {$q * $multiple}]
    }
  }
  #***************************************
  # Report frame
  #***************************************
  snit::widget ReportFrame {
    # Frame components
    component reportRP;#		The report itself
    component searchFrame;#		Search Frame
    component statisticsTF;#		Statistics Title Frame
    component   durationStatCheck;#	Duration Checkbutton
    component   avespeedStatCheck;#	Average Speed Checkbutton
    component   maxspeedStatCheck;#	Maximim Speed Checkbutton
    component   distanceStatCheck;#	Distance Checkbutton
    component button;#			Generate Report Button

    delegate option -viewheight to reportRP as -height
    delegate option -viewwidth to reportRP as -width

    variable _lineNo -1
    variable needPage
    variable curPage
    variable page
    variable _DurationStats no;#	Include duration statistics?
    variable _AveSpeedStats no;#	Include average speed statistics?
    variable _MaxSpeedStats no;#	Include maximum speed statistics?
    variable _DistanceStats no;#	Include distance statistics?
    variable _HaveReport no;#		Flag to indicate that a valid report 
    typevariable _Baselineskip 
    typevariable _HeadFont {}
    typevariable _SubTitleFont {}
    typevariable _BodyFont {}
    typevariable _StatFont {}
    typevariable _FontSize 12
    typeconstructor {
      set _HeadFont [font create -family Times -size $_FontSize -weight bold \
				 -slant roman]
      set _SubTitleFont [font create -family Times -size $_FontSize -weight bold \
				 -slant italic]
      set _BodyFont [font create -family Times -size $_FontSize -weight normal \
				 -slant roman]
      set _StatFont [font create -family Courier -size $_FontSize -weight normal \
				 -slant roman]
      set _Baselineskip [font metrics $_BodyFont -linespace]
    }
    #					is available.
    #***************************************
    # Constructor: populate frame with GUI elememnts.
    #***************************************
    constructor {args} {
      set rh [from args -viewheight 1]
      set rw [from args -viewwidth 1]
      # The Report
      install reportRP using ReportPages::ReportPages $win.reportRP \
		-height $rh -width $rw
      pack $reportRP -expand yes -fill both
      # Search frame
      install searchFrame using ::DatabaseWidgets::SearchFrame \
					$win.searchFrame -includesearch no
      pack $searchFrame -expand yes -fill both
      # Statistics Title Frame
      install statisticsTF using TitleFrame::create $win.statisticsTF \
		-side left -text {Include Statistics for:}
      pack $statisticsTF -fill x
      set statisticsTFfr [$statisticsTF getframe]
      install durationStatCheck using checkbutton \
			$statisticsTFfr.durationStatCheck \
		-text "Duration" -onvalue yes -offvalue no \
		-variable [myvar _DurationStats]
      pack $durationStatCheck -side left -fill x -expand yes
      install avespeedStatCheck using checkbutton \
			$statisticsTFfr.avespeedStatCheck \
		-text "Ave. Speed" -onvalue yes -offvalue no \
		-variable [myvar _AveSpeedStats]
      pack $avespeedStatCheck -side left -fill x -expand yes
      install maxspeedStatCheck using checkbutton \
			$statisticsTFfr.maxspeedStatCheck \
		-text "Max. Speed" -onvalue yes -offvalue no \
		-variable [myvar _MaxSpeedStats]
      pack $maxspeedStatCheck -side left -fill x -expand yes
      install distanceStatCheck using checkbutton \
			$statisticsTFfr.distanceStatCheck \
		-text "Distance" -onvalue yes -offvalue no \
		-variable [myvar _DistanceStats]
      pack $distanceStatCheck -side left -fill x -expand yes
      # Generate report button
      install button using Button::create $win.button \
			-text "Generate Report" \
			-command [mymethod _GenerateReport] \
			-state disabled
      pack $button -fill x
      $self configurelist $args
#      update idle
#      puts stderr "*** $self constructor: requested height of Report frame is [winfo reqheight $reportRP]"
#      puts stderr "*** $self constructor: requested height of search frame is [winfo reqheight $searchFrame]"
#      puts stderr "*** $self constructor: requested height of button is [winfo reqheight $button]"
#      puts stderr "*** $self constructor: total requested height [winfo reqheight $win]"
    }
    #***************************************
    # Method open: connect to a database. Create statement objects to 
    # add records and enable all GUI elements.
    #***************************************
    method open {connection} {
      catch {$self close};#		Close existing connection, if any
      $searchFrame open $connection
      # Enable all of the GUI elememnts
      $button configure -state normal
    }
    #***************************************
    # Method close: drop statement object and disable all GUI elememnts.
    #***************************************
    method close {} {
      # Drop statement object
      $searchFrame close
      # Disable all GUI elememnts
      $button configure -state disabled
    }
    #***************************************
    # Method _GenerateReport -- Generate the report
    #***************************************
    method _Report1Row {row} {
      if {$needPage} {
	incr curPage
	set page [$reportRP addpage $curPage]
	set needPage no
	$self _PageHeading $curPage [lindex $row 0]
      }
      eval [list $self _ReportTrip] $row
      if {[expr {$_lineNo + 10}] > 60} {set needPage yes}
    }
    method _GenerateReport {} {
      set needPage yes
      set curPage 0
      $reportRP deleteallpages
      # Generate the base report
      $searchFrame configure \
        -orderby {starttime} \
	-selectfields {*} \
	-rowfunction [mymethod _Report1Row] \
	-searchresulthook {} \
	-searchstarthook {} \
	-callbackargs {}
      $searchFrame dosearch
      # Generate statistics summary
      if {$_DurationStats || $_AveSpeedStats || 
	  $_MaxSpeedStats || $_DistanceStats} {
	set statsLines 5
	if {$_DurationStats} {incr statsLines}
	if {$_AveSpeedStats} {incr statsLines}
	if {$_MaxSpeedStats} {incr statsLines}
	if {$_DistanceStats} {incr statsLines}
	incr _lineNo
	if {[expr {$_lineNo + $statsLines}] > 60} {
	  incr curPage
	  set page [$reportRP addpage $curPage]
	  $self _PageHeading $curPage {Statistics}
	}
	set x 18p
        set y [expr {$_lineNo * $_Baselineskip}]
	$page create text $x $y -anchor nw \
		-text "[format {%-13s|%10s%10s%10s%10s%10s%10s} Value Min Max Average Sum StdDev Variance]" \
		-font $_StatFont
	incr _lineNo
        set y [expr {$_lineNo * $_Baselineskip}]
	$page create text $x $y -anchor nw \
		-text "[string repeat - 79]" -font $_StatFont
        incr _lineNo
        foreach l {Duration {Ave. Speed} {Max. Speed} Distance} \
		f [list $_DurationStats $_AveSpeedStats $_MaxSpeedStats $_DistanceStats] \
		c {duration/60 averagespeed maxspeed distance} {
	  if {$f} {
	    $searchFrame configure -orderby {} \
	      -selectfields "MIN($c), MAX($c), AVG($c), SUM($c), STDDEV($c), VARIANCE($c)" \
	      -searchresulthook {} -searchstarthook {} -callbackargs [list "$l"] \
	      -rowfunction [mymethod _ReportStatRow]
	    $searchFrame dosearch	    
	  }
	}
      }
      $reportRP gotopage 1
      set _HaveReport yes
    }
    method _ReportStatRow {row label} {
      foreach {min max ave sum stddev variance} $row {
	set x 18p
	set y [expr {$_lineNo * $_Baselineskip}]
	$page create text $x $y -anchor nw \
		-text "[format {%-13s|%10.3f%10.3f%10.3f%10.3f%10.3f%10.3f} \
				"$label" $min $max $ave $sum $stddev $variance]" \
		-font $_StatFont
	incr _lineNo
      }
    }
    method _PageHeading {pageno heading} {
      set _lineNo 3
      set y [expr {$_lineNo * $_Baselineskip}]
      set x 18p
      $page create text $x $y -anchor nw -text "Trip Report" \
			-font $_HeadFont
      set x 594p
      $page create text $x $y -anchor ne \
			-text "$heading  [format {%3d} $pageno]" \
			-font $_HeadFont
      incr _lineNo
      set y [expr {$_lineNo * $_Baselineskip}]
      set x 72p
      set id [$page create text 8i 0 -text . -font $_SubTitleFont]
      set rm [lindex [$page bbox $id] 2]
      $page delete $id
      set whereclause "[$searchFrame whereclause]"
      while {[string length "$whereclause"] > 0} {
	set whereclause1 "$whereclause"
	set lastIndex [string length "$whereclause"]
	set id [$page create text $x $y -anchor nw -text "$whereclause1" \
		 -font $_SubTitleFont]
	set right [lindex [$page bbox $id] 2]
	while {$right > $rm} {
	  set lastIndex [string last { } "$whereclause" $lastIndex]
	  if {$lastIndex <= 0} {break}
	  $page delete $id
	  set whereclause1 [string range "$whereclause" 0 $lastIndex]
	  set id [$page create text $x $y -anchor nw -text "$whereclause1" \
			-font $_SubTitleFont]
	  set right [lindex [$page bbox $id] 2]
	}
	incr _lineNo
	set y [expr {$_lineNo * $_Baselineskip}]
	if {$lastIndex <= 0} {break}
	set whereclause [string trim [string range "$whereclause" $lastIndex end]]
      }
      incr _lineNo
      if {$_lineNo < 6} {set _lineNo 6}
    }
    method _ReportTrip {start end distance maxspeed duration avespeed 
			comments} {
      set x 18p
      set y [expr {$_lineNo * $_Baselineskip}]
      $page create text $x $y -anchor nw \
		-text "Trip at $start, lasting [clock format $duration -format {%H:%M} -gmt yes], until $end" \
		-font $_BodyFont
      incr _lineNo
      set y [expr {$_lineNo * $_Baselineskip}]
      set x 36p
      $page create text $x $y -anchor nw \
		-text "[format {%5.2f} $distance] Miles traveled, speeds: [format {%4.1f} $avespeed] mph average, [format {%4.1f} $maxspeed] mph maximum." \
		-font $_BodyFont
      incr _lineNo
      set y [expr {$_lineNo * $_Baselineskip}]
      regsub -all {[[:space:]]+} "$comments" { } comments
      set x 72p
      set id [$page create text 8i 0 -text . -font $_BodyFont]
      set rm [lindex [$page bbox $id] 2]
      $page delete $id
      while {[string length "$comments"] > 0} {
	set comments1 "$comments"
	set lastIndex [string length "$comments"]
	set id [$page create text $x $y -anchor nw -text "$comments1" \
		-font $_BodyFont]
	set right [lindex [$page bbox $id] 2]
	while {$right > $rm} {
	  set lastIndex [string last { } "$comments" $lastIndex]
	  if  {$lastIndex <= 0} {break}
	  $page delete $id
	  set comments1 [string range "$comments" 0 $lastIndex]
	  set id [$page create text $x $y -anchor nw -text "$comments1" \
		-font $_BodyFont]
	  set right [lindex [$page bbox $id] 2]
	}
	incr _lineNo
        set y [expr {$_lineNo * $_Baselineskip}]
	if {$lastIndex <= 0} {break}
	set comments [string trim [string range "$comments" $lastIndex end]]
      }
      incr _lineNo
    }
    method printP {} {
      return yes
    }
    method print {} {
      if {!$_HaveReport} {return}
      set currentpage [$reportRP currentpage]
      set pageheight  11i
      set pagewidth  8.5i
      set landscape    no
      set lpfp [Printer::OpenPrinter "Print Report" $win pageheight pagewidth landscape]
      if {[string equal "$lpfp" {}]} {return}
      set pages [$reportRP allpages]
      puts $lpfp {%!PS-Adobe-2.0}
      puts $lpfp {%%Creator: TrikeTrips Report Generator $Id$}
      puts $lpfp "%%Title: Trip Reports: [$searchFrame whereclause]"
      puts $lpfp "%%Pages: [llength $pages]"
      puts $lpfp "%%BoundingBox: 0 0 [$type _LengthToPoints $pagewidth] [$type _LengthToPoints $pageheight]"
      puts $lpfp {%%EndComments}
      puts $lpfp "%%BeginProlog"
      puts $lpfp "/EncapDict 200 dict def EncapDict begin"
      puts $lpfp "/showpage {} def /erasepage {} def /copypage {} def end"
      puts $lpfp "/BeginInclude {0 setgray 0 setlinecap 1 setlinewidth"
      puts $lpfp "0 setlinejoin 10 setmiterlimit \[\] 0 setdash"
      puts $lpfp "/languagelevel where {"
      puts $lpfp "  pop"
      puts $lpfp "  languagelevel 2 ge {"
      puts $lpfp "    false setoverprint"
      puts $lpfp "    false setstrokeadjust"
      puts $lpfp "  } if"
      puts $lpfp "} if"
      puts $lpfp "newpath"
      puts $lpfp "save 0 0 translate EncapDict begin} def"
      puts $lpfp "/EndInclude {restore end} def"
      puts $lpfp "%%EndProlog"
      set pn 0
      foreach p $pages {
	incr pn
	puts $lpfp "%%Page: $p $pn"
	$reportRP gotopage $p
	set page [$reportRP getpage $p]
	set postscript "[$page postscript -pageheight $pageheight \
					  -pagewidth $pagewidth -x 0 -y 0 \
					  -height $pageheight \
					  -width $pagewidth -pagex 0 -pagey $pageheight \
					  -pageanchor nw]"
	regsub -all -line {^%%(.*)$} "$postscript" {%#\1} postscript
	puts $lpfp "BeginInclude"
	puts $lpfp "$postscript"
	puts $lpfp "EndInclude showpage"
      }
      puts $lpfp "%%EOF"
      close $lpfp
      $reportRP gotopage $currentpage
    }
    typemethod _LengthToPoints {length} {
      if {[regexp {^([0-9.]+)([cmip])$} "$length"  -> l u] <= 0} {
	return [expr {int($length)}]
      }
      switch $u {
	c {
	  return [expr {int(72  * ($l * 2.54))}]
	}
	m {
	  return [expr {int(72  * ($l * 25.4))}]
	}
	i {
	  return [expr {int(72  * $l)}]
	}
	p {
	  return [expr {int($l)}]
	}
      }
    }
    typemethod _RoundUp {value multiple} {
      set q [expr {int(ceil(double($value) / double($multiple)))}]
      return [expr {$q * $multiple}]
    }
  }    
  #***************************************
  # Export selected rows to a CSV (Comma Separated Value) file
  #***************************************
  snit::widgetadaptor ExportCSVDialog {
    # CSV File types
    typevariable CSVFileTypes {
      {{CSV Files} {.csv} TEXT}
      {{All Files} *      TEXT}
    }
    #*************************************
    # Quote CSV value field
    #*************************************
    typemethod QuoteCSV {string {forceQuote no}} {
      if {[regexp {[,"]} "$string"] > 0} {
	regsub -all {["]} "$string" {""} string
	return "\"$string\""
      } elseif {$forceQuote} {
	return "\"$string\""
      } else {
	return "$string"
      }
    }
    # Additional components:
    component fileLF;#		File name label frame
    component   fileE;#		File name entry
    component   fileB;#		File browse button
    # Instance variable:
    variable _FP {};#		Open file
    # Search frame dialog constructor:
    DatabaseWidgets::SearchDialogConstructor ExportCSVDialog Export {
      # Output file:
      install fileLF using LabelFrame::create [$hull getframe].fileLF \
				-text "Export File:"\
				-width $::DatabaseWidgets::LabelWidth
      pack $fileLF -fill x
      set fileLFfr [$fileLF getframe]
      install fileE using Entry::create $fileLFfr.fileE
      pack $fileE -expand yes -fill x -side left
      install fileB using Button::create $fileLFfr.fileB \
			-text Browse\
			-command [mymethod _Browse]
      pack $fileB -side right
    }
    #***************************************
    # Browse for an output (CSV) file
    #***************************************
    method _Browse {} {
      set existingFile "[$fileE cget -text]"
      set initdir "[file dirname $existingFile]"
      set newfile [tk_getSaveFile -title "File to export to" -parent $win \
			-initialfile "$existingFile" -initialdir "$initdir" \
			-defaultextension {.csv} -filetypes $CSVFileTypes]
      if {[string length "$newfile"] > 0} {
	$fileE configure -text "$newfile"
      }
    }
    #***************************************
    # Unused Draw Hook
    #***************************************
    method _DrawHook {} {}
    #***************************************
    # Start hook: Open output file
    #***************************************
    method _StartHook {} {
      set filename "[$fileE cget -text]"
      if {[catch {open "$filename" w} _FP]} {
	tk_messageBox -type ok -icon error -message "Could not open $filename: $_FP"
	set _FP {}
        return false
      } else {
        return true
      }
    }
    #*************************************
    # Export one CSV row
    #*************************************
    method _RowFunction {row} {
      set needcomma no
      foreach ele $row flag {no no no no no no yes} {
	if {$needcomma} {puts -nonewline $_FP {,}}
        if {$flag} {regsub -all "\n" "$ele" {\n} ele}
	puts -nonewline $_FP [$type QuoteCSV "$ele" $flag]
	set needcomma yes
      }
      puts $_FP {}
    }
    #***************************************
    # Result Hook: close output file
    #***************************************
    method _ResultHook {count} {
      close $_FP
      set _FP {}
      tk_messageBox -type ok -icon info -message "$count rows exported"
    }
  }
  #***************************************
  # Trends Graph Frame
  #***************************************
  snit::widget TrendGraphFrame {

    # Frame components
    component graphSW;#			The graph canvas scrollbars
    component   graphCanvas;#		  The graph canvas
    component xAxisFlagsLF;#		The X Axis flags
    component   xStarttimeRadio;#	  Start flag
    component   xDistanceRadio;#	  Distance flag
    component   xDurationRadio;#	  Duration flag
    component yAxisFlagsLF;#		The Y Axis flags
    component   yDistanceRadio;#	  Distance flag
    component   yDurationRadio;#        Duration flag
    component   yAveSpeedRadio;#	  Average Speed flag
    component searchFrame;#		Search Frame
    component button;#			Draw Trend button

    variable _XAxisFlag starttime;#	X Axis flag
    variable _YAxisFlag distance;#	Y Axis flag
    variable _HaveGraph no;#		Have a graph?
    delegate option -graphheight to graphCanvas as -height
    delegate option -graphwidth to graphCanvas as -width
    #***************************************
    # Constructor: populate frame with GUI elememnts.
    #***************************************
    constructor {args} {
      # Create the drawing area (scrolled canvas)
      install graphSW using ScrolledWindow::create $win.graphSW \
				-scrollbar both -auto both
      pack $graphSW -expand yes -fill both
      install graphCanvas using canvas $graphSW.graphCanvas \
				-background white -relief sunken
      pack $graphCanvas -expand yes -fill both
      $graphSW setwidget $graphCanvas
      # X Axis Flags
      install xAxisFlagsLF using LabelFrame::create $win.xAxisFlagsLF \
				-text "X Axis:" \
				-width $::DatabaseWidgets::LabelWidth \
				-state disabled
      pack $xAxisFlagsLF -fill x
      set xAxisFlagsLFfr [$xAxisFlagsLF getframe]
      foreach name {xStarttimeRadio xDistanceRadio xDurationRadio} \
	      value {starttime distance duration} \
	      text {{Start Time} Distance Duration} {
	install $name using radiobutton $xAxisFlagsLFfr.$name \
		-indicatoron yes -value $value -text "$text" \
		-state disabled -variable [myvar _XAxisFlag]
	pack [set $name] -side left
      }
      # Y Axis Flags
      install yAxisFlagsLF using LabelFrame::create $win.yAxisFlagsLF \
				-text "Y Axis:" \
				-width $::DatabaseWidgets::LabelWidth \
				-state disabled
      pack $yAxisFlagsLF -fill x
      set yAxisFlagsLFfr [$yAxisFlagsLF getframe]
      foreach name {yDistanceRadio yDurationRadio yAveSpeedRadio} \
	      value {distance duration averagespeed} \
	      text {Distance Duration {Average Speed}} {
	install $name using radiobutton $yAxisFlagsLFfr.$name \
		-indicatoron yes -value $value -text "$text" \
		-state disabled -variable [myvar _YAxisFlag]
	pack [set $name] -side left
      }
      # Search frame
      install searchFrame using ::DatabaseWidgets::SearchFrame \
					$win.searchFrame -includesearch no
      pack $searchFrame -expand yes -fill both
      # Draw graph button
      install button using Button::create $win.button \
		-text "Draw Graph" \
		-command [mymethod _DrawGraph] \
		-state disabled
      pack $button -fill x
      $self configurelist $args      
    }

    #***************************************
    # Method open: connect to a database. Create statement objects to 
    # add records and enable all GUI elements.
    #***************************************
    method open {connection} {
      catch {$self close};#             Close existing connection, if any
      $searchFrame open $connection
      # Enable all of the GUI elememnts
      $xAxisFlagsLF configure -state normal
      $xStarttimeRadio configure -state normal
      $xDistanceRadio configure -state normal
      $xDurationRadio configure -state normal
      $yAxisFlagsLF configure -state normal
      $yDistanceRadio configure -state normal
      $yDurationRadio configure -state normal
      $yAveSpeedRadio configure -state normal
      $button configure -state normal
    }      
    #***************************************
    # Method close: drop statement object and disable all GUI elememnts.
    #***************************************
    method close {} {
      $searchFrame close
      # Disable all of the GUI elememnts
      $xAxisFlagsLF configure -state disabled
      $xStarttimeRadio configure -state disabled
      $xDistanceRadio configure -state disabled
      $xDurationRadio configure -state disabled
      $yAxisFlagsLF configure -state disabled
      $yDistanceRadio configure -state disabled
      $yDurationRadio configure -state disabled
      $yAveSpeedRadio configure -state disabled
      $button configure -state disabled
    }      
    method printP {} {
      return yes
    }
    method print {} {
      if {!$_HaveGraph} {return}
      set pageheight 10i
      set pagewidth   8i
      set landscape  yes
      set lpfp [Printer::OpenPrinter "Print Graph" $win pageheight pagewidth landscape]
      if {[string equal "$lpfp" {}]} {return}
      set sr [$graphCanvas cget -scrollregion]
      set height [expr {[lindex $sr 3] - [lindex $sr 1] + 1}]
      set width  [expr {[lindex $sr 2] - [lindex $sr 0] + 1}]
      set y      [lindex $sr 1]
      set x      [lindex $sr 0]
#      puts stderr "*** $self print: sr = $sr, height = $height, width = $width, y = $y x = $x"
      $graphCanvas postscript -channel $lpfp -height $height -width $width \
      			      -x $x -y $y -rotate $landscape \
			      -pageheight $pageheight -pagewidth $pagewidth
      close $lpfp
    }

    variable _XMax;#		X Axis maximum value
    variable _XMin;#		X Axis minimum value
    variable _XCount;#		X Axis count
    variable _XScale;#		X Axis scaling
    variable _XZero;#		X Axis zero point
    variable _XWidth;#		X Axis width
    variable _XIndex -1;#	X Axis index (used when starttime is X axis)
    variable _XValue 0.0;#	Previous X value
    variable _YMax;#		Y Axis maximum value
    variable _YMin;#		Y Axis minimum value
    variable _YScale;#		Y Axis scaling
    typevariable _YZero 400;#	Y Axis zero point
    typevariable _YHeight 400;#	Y Axis height
    variable _YCount;#		Y Axis count
    variable _YValue 0.0;#	Previous Y value
    variable _YTick;#		Y Tick offset
    variable _YLabel;#		Y Label offset

    method _GetRanges {row} {
      set _XMax   [lindex $row 0]
      set _XMin   [lindex $row 1]
      set _XCount [lindex $row 2]
      set _YMax   [lindex $row 3]
      set _YMin   [lindex $row 4]
      set _YCount [lindex $row 5]
    }
    method _DrawGraph {} {
      set selection     "MAX($_XAxisFlag), MIN($_XAxisFlag), COUNT($_XAxisFlag),"
      append selection " MAX($_YAxisFlag), MIN($_YAxisFlag), COUNT($_YAxisFlag)"
      $searchFrame configure \
	-orderby {} \
	-selectfields "$selection" \
	-rowfunction [mymethod _GetRanges] \
	-searchresulthook {} \
	-searchstarthook {} \
	-callbackargs {}
      $searchFrame dosearch
#      puts stderr "*** $self _DrawGraph: _XMax = $_XMax, _XMin = $_XMin, _XCount = $_XCount, _YMax = $_YMax, _YMin = $_YMin, _YCount = $_YCount"
      if {$_XCount == 0 || $_YCount == 0} {
	tk_messageBox -type ok -icon warning -message "No records matched!"
	return
      }
      set selection "$_XAxisFlag, $_YAxisFlag"
      $searchFrame configure \
	-orderby "$_XAxisFlag" \
	-selectfields "$selection" \
	-rowfunction [mymethod _DrawGraphPoint] \
	-searchresulthook [mymethod _FinishGraph] \
	-searchstarthook [mymethod _StartGraph] \
	-callbackargs {}
      $searchFrame dosearch
    }
    method _StartGraph {} {
      set _HaveGraph no
      set _XValue -999
      set _YValue -999
      switch $_XAxisFlag {
	starttime {
	  set _XMin 0.5
	  set _XMax $_XCount
	  set _XIndex 0
	  set tempId [$graphCanvas create text 0 0 -anchor nw -text [clock format 0 -format "%b %d, %Y\n%H:%M"]]
	  set dtbbox [$graphCanvas bbox $tempId]
	  set x1W    [expr {[lindex $dtbbox 2] + 30}]
	  set _XWidth [expr {$x1W * $_XCount}]
	  set _XScale $x1W
	}
	distance {
	  set _XMin [$type _RoundDown $_XMin 5]
	  set _XMax [$type _RoundUp   $_XMax 5]
	  set tempId [$graphCanvas create text 0 0 -anchor nw -text "[format {%5.2fmi} 0.0]"]
	  set lwidth [expr {[lindex [$graphCanvas bbox $tempId] 2] + 20}]
	  set xrange [expr {$_XMax - $_XMin}]
	  set _XWidth [expr {$lwidth * 10}]
	  set _XScale [expr {double($_XWidth) / double($xrange)}]
	}
	duration {
	  set _XMin [$type _RoundDown $_XMin 600]
	  set _XMax [$type _RoundUp   $_XMax 600]
	  set tempId [$graphCanvas create text 0 0 -anchor nw -text "[clock format 0 -format {%H:%M} -gmt yes]"]
	  set lwidth [expr {[lindex [$graphCanvas bbox $tempId] 2] + 20}]
	  set xrange [expr {$_XMax - $_XMin}]
	  set _XWidth [expr {$lwidth * 10}]
	  set _XScale [expr {double($_XWidth) / double($xrange)}]
	}
      }
      switch $_YAxisFlag {
	distance {
	  set _YMin [$type _RoundDown $_YMin 5]
	  set _YMax [$type _RoundUp   $_YMax 5]
	  set tempId [$graphCanvas create text 0 0 -anchor nw -text "[format {%5.2f} 0.0]"]
	  set _XZero  [expr {[lindex [$graphCanvas bbox $tempId] 2] + 30}]
	}
	duration {
	  set _YMin [$type _RoundDown $_YMin 600]
	  set _YMax [$type _RoundUp   $_YMax 600]
	  set tempId [$graphCanvas create text 0 0 -anchor nw -text "[clock format 0 -format {%H:%M} -gmt yes]"]
	  set _XZero  [expr {[lindex [$graphCanvas bbox $tempId] 2] + 30}]
	}
	averagespeed {
	  set _YMin [$type _RoundDown $_YMin 1]
	  set _YMax [$type _RoundUp   $_YMax 1]
	  set tempId [$graphCanvas create text 0 0 -anchor nw -text "[format {%4.1f} 0.0]"]
	  set _XZero  [expr {[lindex [$graphCanvas bbox $tempId] 2] + 30}]
	}
      }
      set yrange [expr {$_YMax - $_YMin}]
      set _YScale [expr {double($_YHeight) / double($yrange)}]
      set xt [expr {$_XZero - 20}]
      set xl [expr {$_XZero - 25}]
      $graphCanvas delete all
      foreach sp {0 .1 .2 .3 .4 .5 .6 .7 .8 .9 1.0} {
	set tick [expr {$sp * $_YHeight}]
	set value [expr {$_YMin + ($sp * $yrange)}]
	set y [expr {$_YZero - $tick}]
	switch $_YAxisFlag {
	  distance {
	    $graphCanvas create text $xl $y -anchor e -text "[format {%5.2f} $value]"
	  }
	  duration {
	    $graphCanvas create text $xl $y -anchor e -text "[clock format [expr {int($value)}] -format {%H:%M} -gmt yes]"
	  }
	  averagespeed {
	    $graphCanvas create text $xl $y -anchor e -text "[format {%4.1f} $value]"
	  }
	}
	$graphCanvas create line $xt $y $_XZero $y
	$graphCanvas create line $_XZero $y [expr {$_XZero + $_XWidth}] $y -dash -
      }
      switch $_YAxisFlag {
        distance {
	  $graphCanvas create text $xl [expr {$_YZero + 10}] -text "Miles" -anchor ne
	}
	duration {
	  $graphCanvas create text $xl [expr {$_YZero + 10}] -text "Time" -anchor ne
	}
	averagespeed {
	  $graphCanvas create text $xl [expr {$_YZero + 10}] -text "MPH" -anchor ne
	}
      }
      $graphCanvas create line $_XZero $_YZero $_XZero [expr {$_YZero - $_YHeight}]
      $graphCanvas create line $_XZero $_YZero [expr {$_XZero + $_XWidth}] $_YZero
      set _YTick [expr {$_YZero + 20}]
      set _YLabel [expr {$_YZero + 25}]
      if {![string equal "$_XAxisFlag" starttime]} {
	foreach sp {0 .1 .2 .3 .4 .5 .6 .7 .8 .9 1.0} {
	  set tick [expr {$sp * $_XWidth}]
	  set value [expr {$_XMin + ($sp * $xrange)}]
	  set x [expr {$_XZero + $tick}]
	  switch $_XAxisFlag {
	    distance {
	      $graphCanvas create text $x $_YLabel -anchor n -text "[format {%5.2fmi} $value]"
	    }
	    duration {
	      $graphCanvas create text $x $_YLabel -anchor n -text "[clock format [expr {int($value)}] -format {%H:%M} -gmt yes]"
	    }
	  }
	  $graphCanvas create line $x $_YTick $x $_YZero
	}
      }
      return true
    }
    method _DrawGraphPoint {row} {
      if {[string equal "$_XAxisFlag" starttime]} {
	incr _XIndex
	set startsecs [clock scan "[lindex $row 0]"]
	set label [clock format $startsecs -format "%b %d, %Y\n%H:%M"]
	set xoff [expr {($_XIndex - $_XMin) * $_XScale}]
	set x [expr {$xoff + $_XZero}]
	$graphCanvas create text $x $_YLabel -anchor n -text "$label"
	$graphCanvas create line $x $_YTick $x $_YZero
      } else {
	set xoff [expr {([lindex $row 0] - $_XMin) * $_XScale}]
	set x [expr {$xoff + $_XZero}]
      }
      set yoff [expr {([lindex $row 1] - $_YMin) * $_YScale}]
      set y [expr {$_YZero - $yoff}]
      if {$_XValue != -999 && $_YValue != -999} {
	$graphCanvas create line $_XValue $_YValue $x $y -width 2
      }
      $graphCanvas create oval [expr {$x - 2}] [expr {$y - 2}] \
			       [expr {$x + 2}] [expr {$y + 2}] -fill red
      set _XValue $x
      set _YValue $y
    }
    method _FinishGraph {rowcount} {
      set bbox [$graphCanvas bbox all]
      set bottom [expr {[lindex $bbox 3] + 10}]
      set center [expr {([lindex $bbox 0] + [lindex $bbox 2]) / 2.0}]
      $graphCanvas create text $center $bottom -anchor n -text "$_XAxisFlag vs. $_YAxisFlag"
      $graphCanvas configure -scrollregion [$graphCanvas bbox all]
      set _HaveGraph yes
      return true
    }
    typemethod _RoundUp {value multiple} {
      set q [expr {int(ceil(double($value) / double($multiple)))}]
      return [expr {$q * $multiple}]
    }
    typemethod _RoundDown {value multiple} {
      set q [expr {int(floor(double($value) / double($multiple)))}]
      return [expr {$q * $multiple}]
    }
  }
}

#***************************************
# (Re-)Initialize datebase: drop and rebuild Trip Table
#***************************************
proc DatabaseWidgets::CreateTripTable {Connection} {
  variable TableName
  variable TableSpecification

  # Check for existing table
  if {[catch {$Connection tables $TableName} tableinfo]} {
    tk_messageBox -type ok -icon error -message "Open a connection to a database first!"
    return
  }
  # Table already exists?  Verify clobbering it...
  if {[llength $tableinfo] > 0} {
    if {[string equal [tk_messageBox -type yesno -default no -icon question \
	-message "Are you sure you want to re-initialize your database?"] {no}]} {
      return
    }
    $Connection "drop table $TableName"
  }
  # Create trip table.
  set result [catch {$Connection "create TABLE $TableName $TableSpecification"} error]
  if {$result} {tk_messageBox -type ok -icon error -message "$error"}
}

proc DatabaseWidgets::ImportCSVFile {filename connection} {
  variable TableName

  # Make sure we have a table to add rows to.
  if {[catch {$connection tables $TableName} tableinfo]} {
#  puts stderr "*** DatabaseWidgets::ImportCSVFile: tableinfo = $tableinfo"
    tk_messageBox -type ok -icon error -message "Please initialize the database first!"
    return
  }
  if {[catch {open "$filename" r} infp]} {
    tk_messageBox -type ok -icon error -message "filename: $infp"
    return
  }
  # Create statement object
  set inserttrip \
	[$connection \
		statement [statementID create] \
		"insert into $TableName values (?,?,?,?,?,?,?)" \
		{TIMESTAMP TIMESTAMP REAL REAL INTEGER REAL CHAR}]
  set count 0
  while {[gets $infp line] >= 0} {
    set res [catch {$inserttrip run [csv::split "$line"]} error]
    if {$res} {
      close $infp
      catch {$inserttrip drop}
      tk_messageBox -type ok -icon error -message "$error,\n$count records imported"
      return
    }
    incr count
  }
  close $infp
  catch {$inserttrip drop}
  tk_messageBox -type ok -icon info -message "$count records imported"
}

package provide DatabaseWidgets 1.0
 
