#* 
#* ------------------------------------------------------------------
#* TrikeTrips.tcl - Trike Trips -- store, compute, and display trike trip info.
#* Created by Robert Heller on Thu May 24 09:14:34 2007
#* ------------------------------------------------------------------
#* Modification History: $Log: headerfile.text,v $
#* Modification History: Revision 1.1  2002/07/28 14:03:50  heller
#* Modification History: Add it copyright notice headers
#* Modification History:
#* ------------------------------------------------------------------
#* Contents:
#* ------------------------------------------------------------------
#*  
#*     Generic Project
#*     Copyright (C) 2005  Robert Heller D/B/A Deepwoods Software
#* 			51 Locke Hill Road
#* 			Wendell, MA 01379-9728
#* 
#*     This program is free software; you can redistribute it and/or modify
#*     it under the terms of the GNU General Public License as published by
#*     the Free Software Foundation; either version 2 of the License, or
#*     (at your option) any later version.
#* 
#*     This program is distributed in the hope that it will be useful,
#*     but WITHOUT ANY WARRANTY; without even the implied warranty of
#*     MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
#*     GNU General Public License for more details.
#* 
#*     You should have received a copy of the GNU General Public License
#*     along with this program; if not, write to the Free Software
#*     Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
#* 
#*  
#* 

#       Load dict extension, if Tcl version is less than 8.5
if {[package vcompare [package present Tcl] 8.5] < 0} {
  package require dict
}

# Load other packages:
package require snodbc;#                DB interface code (tclodbc compatible)
package require Tk;#                    Base GUI Toolkit
package require BWidget;#               BWidget toolkit
package require BWHelp;#                BWidget help package
package require BWStdMenuBar;#          BWidget menu bar package
package require snit;#                  Snit package
package require SelectDatasource;#      Data source selector dialog
package require DatabaseWidgets;#	Database objects


# Image Directory setup
global ImageDir 
set ImageDir [file join [file dirname [file dirname [info script]]] \
                        Common]
# Help Directory setup
global HelpDir
set HelpDir [file join [file dirname [file dirname [file dirname \
                                                        [info script]]]] Help]

namespace eval TrikeTrips  {#dummy}

#*************************************
# Proc bound to tab click -- check for a printable page
#*************************************
proc TrikeTrips::TabClick {page} {
  variable TabMap
  variable Tools
  variable Main
#  puts stderr "*** TrikeTrips::TabClick $page"
  if {[catch {set TabMap($page)} tabFrame]} {
    $Main setmenustate needprintable disabled
    $Tools.print configure -state disabled
    return
  }
  if {[catch {$tabFrame printP} printable]} {
    $Main setmenustate needprintable disabled
    $Tools.print configure -state disabled
    return
  }
  if {$printable} {
    $Main setmenustate needprintable normal
    $Tools.print configure -state normal
  } else {
    $Main setmenustate needprintable disabled
    $Tools.print configure -state disabled
  }
}

# Main program namespace.
namespace eval TrikeTrips {
  #		Dummy type to generate unique conn IDs
  snit::type connectionID {
    pragma -hasinstances false
    typevariable id 0
    typemethod create {} {
      incr id
      return "${type}${id}"
    }
  }
  variable Connection {};#		Current connection ID
  # Create a standard menu.
  variable Menu [StdMenuBar::MakeMenu \
	 -file {"&File" {file} {file} 0 {
		{command "&New" {file:new} "Create New Database"  {Ctrl n} \
			-command {TrikeTrips::NewDatabase} -state disabled}
		{command "&Initialize DB" {file:init needconnection} \
			"(Re-)Initialize current DB" \
			{Ctrl i} -command {TrikeTrips::InitDatabase} \
			-state disabled}
		{command "&Open..." {file:open} "Connect to an existing DB" \
			{Ctrl o} -command {TrikeTrips::OpenDatabase}}
		{command "&Save"    {file:save} "" {} -state disabled}
		{command "Save &As..."    {file:save} "" {} -state disabled}
		{command "&Import..." {file:import needconnection} \
			"Import trip data" {Alt i} \
			-command {TrikeTrips::ImportData} -state disabled}
		{cascad "&Export"  {file:export needconnection} {file:export} 0 {
			{command "As &CSV..." {file:export:csv needconnection}
				 "Export as CSV" {CtrlAlt c}
				 -command {TrikeTrips::Export::ExportCSV} \
				 -state disabled}
		  }
		}
		{command "&Print..." {file:print needprintable} "Print Trip Report" {Ctrl p} \
			-command {TrikeTrips::Print} \
			-state disabled}
		{command "&Close" {file:close} "Close the application" \
			{Ctrl q} -command {TrikeTrips::CareFulExit}}
		{command "E&xit" {file:exit} "Close the application" {Ctrl q} \
			-command {TrikeTrips::CareFulExit}}
                }
  }]
  # Window manager configurations
  wm positionfrom . user
  wm sizefrom . ""
  wm maxsize . 1265 994
  wm minsize . 1 1
  wm protocol . WM_DELETE_WINDOW {TrikeTrips::CareFulExit}
  wm title . {Trike Trip Database Tk}
  wm withdraw .

  variable Status {};#		Status variable
  # Main window
  variable Main [MainFrame::create .main -menu $Menu \
					 -textvariable TrikeTrips::Status]

  pack $Main  -expand yes  -fill both
  $Main showstatusbar status
  # Build toolbar
  variable Tools [$Main addtoolbar]
  foreach gif {new.gif open.gif print.gif csv.gif close.gif} \
	  alt {"Create New Database" "Connect to an existing DB" 
	       "Print Trip Report" "Export to CSV" "Close the application"} \
	  inistate {disabled normal disabled disabled normal} \
	  command {TrikeTrips::NewDatabase TrikeTrips::OpenDatabase 
		   TrikeTrips::Print 
		   TrikeTrips::Export::ExportENV 
		   TrikeTrips::CareFulExit} \
	  name {new open print csv close} \
	  side {left left left left right} {
    image create photo ::TrikeTrips::$name -file [file join $::ImageDir $gif]
    pack [Button::create $Tools.$name -image ::TrikeTrips::$name \
    				      -helptext "$alt" -helptype variable \
				      -helpvar TrikeTrips::Status \
				      -name $name -command $command \
				      -state $inistate] -side $side
  }
  set mf [$Main getframe]

  # Notebook containing several pages.
  variable MainNotebook [NoteBook::create $mf.notebook -side top -homogeneous no]
  pack $MainNotebook -expand yes  -fill both
  $MainNotebook bindtabs <1> TrikeTrips::TabClick
  variable TabMap

  #***************************************************
  # Add trip page
  variable AddTripPath [$MainNotebook insert end addTripTab -text "Add Trip"]
  variable AddFrame [DatabaseWidgets::AddFrame $AddTripPath.addFrame]
  pack $AddFrame -fill both -expand yes
  set TabMap(addTripTab) $AddFrame
  #***************************************************

  #***************************************************
  # Add bar chart page
  # allow for 'short' screens
  set ch [expr {[winfo screenheight $MainNotebook] - 475}]
  if {$ch > 300} {set ch 300}
  variable BarChartPath [$MainNotebook insert end barChartTab -text "Bar Chart"]
  variable BarChartFrame [DatabaseWidgets::BarChartFrame \
			$BarChartPath.barChart -chartheight $ch]
  pack $BarChartFrame -fill both -expand yes
  set TabMap(barChartTab) $BarChartFrame
  #***************************************************

  #***************************************************
  # Add Report page
  # allow for 'short' screens
  set vh [expr {[winfo screenheight $MainNotebook] - 550}]
  if {$vh > 400} {set vh 400}
  set vw [expr {int($vh * .75)}]
#  puts stderr "*** vh = $vh"
  variable ReportPath [$MainNotebook insert end reportTab -text "Reports"]
  variable ReportFrame [DatabaseWidgets::ReportFrame $ReportPath.report \
				-viewheight $vh -viewwidth $vw]
  pack $ReportFrame -fill both -expand yes
  set TabMap(reportTab) $ReportFrame
  #***************************************************

  #***************************************************
  # Add Trend graph page
  set gh [expr {[winfo screenheight $MainNotebook] - 500}]
  if {$gh > 300} {set gh 300}
#  puts stderr "*** gh = $gh"
  variable TrendGraphPath [$MainNotebook insert end trendGraphTab -text "Trend Graph"]
  variable TrendGraphFrame [DatabaseWidgets::TrendGraphFrame \
		$TrendGraphPath.trendGraph -graphheight $gh]
  pack $TrendGraphFrame -fill both -expand yes
  set TabMap(trendGraphTab) $TrendGraphFrame
  #***************************************************


  $MainNotebook compute_size;#		Update sizes
  $MainNotebook raise addTripTab;#	Make the add trip page visible
  TabClick addTripTab;#			Invoke tab handler

  # Center window on the screen and map it.
  update idletasks
  set w [winfo toplevel $Main]
  set x [expr {[winfo screenwidth $w]/2 - [winfo reqwidth $w]/2}]
  set y [expr {[winfo screenheight $w]/2 - [winfo reqheight $w]/2}]
  # Make sure that the window is on the screen and set the maximum
  # size of the window is the size of the screen.
  if {$x < 0} {
    set x 0
  }
  if {$y < 0} {
    set y 0
  }
  wm maxsize $w [winfo screenwidth $w] [winfo screenheight $w]
  wm geometry $w +$x+$y
  wm deiconify $w

  # Export selection dialog
  namespace eval Export {
    # Export Dialog
    variable ExportCSVDialog {};#	Export to CSV
  }
}

#*************************************
# Print function
#*************************************
proc TrikeTrips::Print {} {
  variable MainNotebook
  variable TabMap

  set page [$MainNotebook raise]
  if {[catch {set TabMap($page)} tabFrame]} {return}
  if {[catch {$tabFrame printP} printable]} {return}
  if {$printable} {$tabFrame print}
}

#*************************************
# Careful exit function.
#*************************************
proc TrikeTrips::CareFulExit {} {
  if {[string compare \
        [tk_messageBox -default no -icon question -message {Really Quit?} \
                -title {Careful Exit} -type yesno] {yes}] == 0} {
    # Close the database
    CloseDatabase
#    puts stderr "*** TrikeTrips::CareFulExit: CloseDatabase returned"
    # And exit
    exit
  }
}

#*************************************
# Close the database connection.
#*************************************
proc TrikeTrips::CloseDatabase {} {
  variable Main
  variable Tools
  variable Connection
  variable TabMap
  variable Export::ExportCSVDialog

  # Drop all database statement objects.
  foreach t [array names TabMap] {
    catch {$TabMap($t) close}
  }
  catch {$Export::ExportCSVDialog close}
  # Disconnect from database server
  catch {$Connection disconnect}
#  puts stderr "*** TrikeTrips::CloseDatabase $Connection disconnected"
  # Disable GUI elemements
  $Main setmenustate needconnection disabled
  $Tools.print configure -state disabled
  $Tools.csv configure -state disabled
}

#*************************************
# Create a new database
#*************************************
proc TrikeTrips::NewDatabase {} {
  error "Not Implemented yet."
}


#*************************************
# Open a fresh database connection
#*************************************
proc TrikeTrips::OpenDatabase {{connectionString {}}} {
  variable Connection
  variable Main
  variable Tools
  variable TabMap
  variable Export::ExportCSVDialog

  if {[string equal "$connectionString" {}]} {
    # Get the connection string
    set connectionString [SelectDatasource::SelectDatasourceDialog draw \
								-parent $Main]
#    puts stderr "*** TrikeTrips::OpenDatabase: connectionString = '$connectionString'"
  }
  if {[string equal "$connectionString" {}]} {return}
  # Close existing database connection (if any)
  CloseDatabase
  # Open a connection
  set Connection [database connect [connectionID create]  "$connectionString"]
#  puts stderr "*** TrikeTrips::OpenDatabase: connectionString = $connectionString"
  # Activate buttons
  $Main setmenustate needconnection normal
  $Tools.csv configure -state normal
  # Open frames
  foreach t [array names TabMap] {
    $TabMap($t) open $Connection
  }
  # Connect search dialogs.
  catch {$Export::ExportCSVDialog open $Connection}
}


#*************************************
# Initialize a database
#*************************************
proc TrikeTrips::InitDatabase {} {
  variable Connection

  DatabaseWidgets::CreateTripTable $Connection
}

#*************************************
# Export database to CSV
#*************************************
proc TrikeTrips::Export::ExportCSV {} {
  variable ExportCSVDialog

  # Create Export dialog
  if {[string equal "$ExportCSVDialog" {}]} {
    set ExportCSVDialog [DatabaseWidgets::ExportCSVDialog .exportCSV%AUTO%]
    $ExportCSVDialog open $::TrikeTrips::Connection
  }
  # Export rows
  $ExportCSVDialog draw -parent $::TrikeTrips::Main \
			-title "Export selected rows to CSV"
}

proc TrikeTrips::ImportData {{filename {}}} {
  variable Main
  variable Connection

  if {[string equal "$filename" {}]} {
    set filename [tk_getOpenFile -title "CSV File to import" -parent $Main \
				 -initialfile trips.csv -initialdir [pwd] \
				 -defaultextension {.csv} \
				 -filetypes {
					      {{CSV Files} {.csv} TEXT}
					      {{All Files} *      TEXT}
    }]
  }
  if {[string equal "$filename" {}]} {return}
  DatabaseWidgets::ImportCSVFile "$filename" $Connection
}

global argv argv0 argc

if {$argc > 0} {
  TrikeTrips::OpenDatabase "[lindex $argv 0]"
}
