#* 
#* ------------------------------------------------------------------
#* DirectMailTk.tcl - DirectMail DB program
#* Created by Robert Heller on Mon Feb 26 09:26:00 2007
#* ------------------------------------------------------------------
#* Modification History: $Log: headerfile.text,v $
#* Modification History: Revision 1.1  2002/07/28 14:03:50  heller
#* Modification History: Add it copyright notice headers
#* Modification History:
#* ------------------------------------------------------------------
#* Contents:
#* ------------------------------------------------------------------
#*  
#*     Generic Project
#*     Copyright (C) 2005  Robert Heller D/B/A Deepwoods Software
#* 			51 Locke Hill Road
#* 			Wendell, MA 01379-9728
#* 
#*     This program is free software; you can redistribute it and/or modify
#*     it under the terms of the GNU General Public License as published by
#*     the Free Software Foundation; either version 2 of the License, or
#*     (at your option) any later version.
#* 
#*     This program is distributed in the hope that it will be useful,
#*     but WITHOUT ANY WARRANTY; without even the implied warranty of
#*     MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
#*     GNU General Public License for more details.
#* 
#*     You should have received a copy of the GNU General Public License
#*     along with this program; if not, write to the Free Software
#*     Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
#* 
#*  
#* 

#***********************************************************************
#*                                                                     *
#* Main program and direct support code.                               *
#*                                                                     *
#* Create the main GUI window and provide the function callbacks for   *
#* the active GUI elemements (menu items, buttons, etc.).              *
#*                                                                     *
#***********************************************************************

#	Load dict extension, if Tcl version is less than 8.5
if {[package vcompare [package present Tcl] 8.5] < 0} {
  package require dict
}

# Load other packages:
package require snodbc;#		DB interface code (tclodbc compatible)
package require Tk;#			Base GUI Toolkit
package require BWidget;#		BWidget toolkit
package require BWHelp;#		BWidget help package
package require BWStdMenuBar;#		BWidget menu bar package
package require snit;#			Snit package
package require SelectDatasource;#	Data source selector dialog
package require DatabaseWidgets;#	Database objects
package require RecordPage;#		Record display code.
package require PrintLabels;#		Label Printing package

# Image Directory setup
global ImageDir 
set ImageDir [file join [file dirname [file dirname [info script]]] \
                        Common]
# Help Directory setup
global HelpDir
set HelpDir [file join [file dirname [file dirname [file dirname \
                                                        [info script]]]] Help]
# Main program namespace.
namespace eval DirectMailTk {
  #		Dummy type to generate unique conn IDs
  snit::type connectionID {
    pragma -hasinstances false
    typevariable id 0
    typemethod create {} {
      incr id
      return "${type}${id}"
    }
  }
  variable Connection {};#		Current connection ID
  # Create a standard menu.
  variable Menu [StdMenuBar::MakeMenu \
	 -file {"&File" {file} {file} 0 {
		{command "&New" {file:new} "Create New Database"  {Ctrl n} \
			-command {DirectMailTk::NewDatabase} -state disabled}
		{command "&Initialize DB" {file:init needconnection} "(Re-)Initialize current DB" \
			{Ctrl i} -command {DirectMailTk::InitDatabase} \
			-state disabled}
		{command "&Open..." {file:open} "Connect to an existing DB" \
			{Ctrl o} -command {DirectMailTk::OpenDatabase}}
		{command "&Save"    {file:save} "" {} -state disabled}
		{command "Save &As..."    {file:save} "" {} -state disabled}
		{cascad "&Export"  {file:export needconnection} {file:export} 0 {
			{command "As &CSV..." {file:export:csv needconnection}
				 "Export as CSV" {CtrlAlt c}
				 -command {DirectMailTk::Export::ExportCSV} \
				 -state disabled}
			{command "As LaTex &ENVs" {file:export:LaTexENV needconnection}
				 "Export as LaTeX Envelopes (env.sty)"
				 {CtrlAlt e} -command {DirectMailTk::Export::ExportENV} \
				 -state disabled}
			{command "As LaTeX &Letters" {file:export:LaTexLetter needconnection}
				 "Export as LaTeX Letters" {CtrlAlt l} 
				 -command {DirectMailTk::Export::ExportLetter} \
				 -state disabled}
		  }
		}
		{command "&Print..." {file:print needconnection} "Print Labels" {Ctrl p} \
			-command {DirectMailTk::PrintLabels::PrintDatabase} \
			-state disabled}
		{command "&Close" {file:close} "Close the application" \
			{Ctrl q} -command {DirectMailTk::CareFulExit}}
		{command "E&xit" {file:exit} "Close the application" {Ctrl q} \
			-command {DirectMailTk::CareFulExit}}
                }
             }]
  # Window manager configurations
  wm positionfrom . user
  wm sizefrom . ""
  wm maxsize . 1265 994
  wm minsize . 1 1
  wm protocol . WM_DELETE_WINDOW {DirectMailTk::CareFulExit}
  wm title . {Direct Mail Tk}
  wm withdraw .

  variable Status {};#		Status variable
  # Main window
  variable Main [MainFrame::create .main -menu $Menu \
					 -textvariable DirectMailTk::Status]

  pack $Main  -expand yes  -fill both
  $Main showstatusbar status
  # Build toolbar
  variable Tools [$Main addtoolbar]
  foreach gif {new.gif open.gif print.gif mail.gif tex.gif csv.gif close.gif} \
	  alt {"Create New Database" "Connect to an existing DB" 
	       "Print Labels" "Export to Envelopes" "Export to LaTeX Letters"
	       "Export to CSV" "Close the application"} \
	  inistate {disabled normal disabled disabled disabled disabled normal} \
	  command {DirectMailTk::NewDatabase DirectMailTk::OpenDatabase 
		   DirectMailTk::PrintLabels::PrintDatabase 
		   DirectMailTk::Export::ExportENV 
		   DirectMailTk::Export::ExportLetter
		   DirectMailTk::Export::ExportCSV
		   DirectMailTk::CareFulExit} \
	  name {new open print mail tex csv close} \
	  side {left left left left left left right} {
    image create photo ::DirectMailTk::$name -file [file join $::ImageDir $gif]
    pack [Button::create $Tools.$name -image ::DirectMailTk::$name \
    				      -helptext "$alt" -helptype variable \
				      -helpvar DirectMailTk::Status \
				      -name $name -command $command \
				      -state $inistate] -side $side
  }
  set mf [$Main getframe]
  # Notebook containing two pages.
  variable MainNotebook [NoteBook::create $mf.notebook -side top -homogeneous no]
  pack $MainNotebook -expand yes  -fill both

  # Add record page
  variable AddRecordPath [$MainNotebook insert end addRecordTab \
					-text "Add Record"]

  variable AddFrame [DatabaseWidgets::AddFrame $AddRecordPath.addFrame]
  pack $AddFrame -fill both -expand yes
  # Search page
  variable SearchEditPath [$MainNotebook insert end searchEditTab \
					-text "Search and Edit Database"]
  variable SearchFrame [DatabaseWidgets::SearchFrame \
				$SearchEditPath.searchFrame \
				-searchstarthook DirectMailTk::SearchBegin \
				-rowfunction DirectMailTk::SearchRow \
				-searchresulthook DirectMailTk::SearchResults]
  pack $SearchFrame -fill both -expand yes
  set resultsw [ScrolledWindow::create $DirectMailTk::SearchEditPath.results \
			-auto both -scrollbar both]
  pack $resultsw -fill both -expand yes
  variable Results [ListBox::create [$resultsw getframe].list \
					-selectfill yes -selectmode multiple]
  pack $Results -fill both -expand yes
  $resultsw setwidget $Results
  variable ResultButtons [ButtonBox::create $DirectMailTk::SearchEditPath.buttons \
			-homogeneous yes -orient horizontal]
  pack $ResultButtons -fill x
  $ResultButtons add -name delete -text Delete \
				    -command {DirectMailTk::DeleteRecords} \
				    -state disabled
  $ResultButtons add -name view -text View \
				    -command {DirectMailTk::ViewRecords} \
				    -state disabled
  $ResultButtons add -name edit -text Edit \
				    -command {DirectMailTk::EditRecords} \
				    -state disabled
  $ResultButtons configure -state disabled
  
  $MainNotebook compute_size;#		Update sizes
  $MainNotebook raise addRecordTab;#	Make the add record page visible

  # Center window on the screen and map it.
  update idle
  set w [winfo toplevel $Main]
  set x [expr {[winfo screenwidth $w]/2 - [winfo reqwidth $w]/2}]
  set y [expr {[winfo screenheight $w]/2 - [winfo reqheight $w]/2}]
  # Make sure that the window is on the screen and set the maximum
  # size of the window is the size of the screen.
  if {$x < 0} {
    set x 0
  }
  if {$y < 0} {
    set y 0
  }
  wm maxsize $w [winfo screenwidth $w] [winfo screenheight $w]
  wm geom $w +$x+$y
  wm deiconify $w

  # Print and export selection dialogs  
  namespace eval PrintLabels {
    # Print Labels Dialog
    variable PrintLabelsDialog {}
  }
  namespace eval Export {
    # Export Dialogs
    variable ExportCSVDialog {};#	Export to CSV
    variable ExportEnvDialog {};#	Export to LaTeX Envelopes
    variable ExportLetterDialog {};#	Export to LaTeX Letters
  }
}

#*************************************
# Careful exit function.
#*************************************
proc DirectMailTk::CareFulExit {} {
  if {[string compare \
        [tk_messageBox -default no -icon question -message {Really Quit?} \
                -title {Careful Exit} -type yesno] {yes}] == 0} {
    # Close the database
    CloseDatabase
#    puts stderr "*** DirectMailTk::CareFulExit: CloseDatabase returned"
    # And exit
    exit
  }
}

#*************************************
# Close the database connection.
#*************************************
proc DirectMailTk::CloseDatabase {} {
  variable Main
  variable Tools
  variable Connection
  variable AddFrame
  variable SearchFrame
  variable ResultButtons
  # Drop all database statement objects.
  catch {$AddFrame close}
  catch {$SearchFrame close}
  catch {$PrintLabels::PrintLabelsDialog close}
  catch {$Export::ExportCSVDialog close}
  catch {$Export::ExportEnvDialog close}
  catch {$Export::ExportLetterDialog close}
  RecordPage::RecordPage closeallpages
  # Disconnect from database server
  catch {$Connection disconnect}
#  puts stderr "*** DirectMailTk::CloseDatabase $Connection disconnected"
  # Disable GUI elemements
  $ResultButtons configure -state disabled
  $Main setmenustate needconnection disabled
  $Tools.print configure -state disabled
  $Tools.mail configure -state disabled
  $Tools.tex configure -state disabled
  $Tools.csv configure -state disabled
}

#*************************************
# Create a new database
#*************************************
proc DirectMailTk::NewDatabase {} {
  error "Not Implemented yet."
}


#*************************************
# Open a fresh database connection
#*************************************
proc DirectMailTk::OpenDatabase {} {
  variable Connection
  variable Main
  variable Tools
  variable AddFrame
  variable SearchFrame

  # Get the connection string
  set connectionString [SelectDatasource::SelectDatasourceDialog draw \
								-parent $Main]
  if {[string equal "$connectionString" {}]} {return}
  # Close existing database connection (if any)
  CloseDatabase
  # Open a connection
  set Connection [database connect [connectionID create]  "$connectionString"]
#  puts stderr "*** DirectMailTk::OpenDatabase: connectionString = $connectionString"
  # Activate buttons
  $Main setmenustate needconnection normal
  $Tools.print configure -state normal
  $Tools.mail configure -state normal
  $Tools.tex configure -state normal
  $Tools.csv configure -state normal
  # Open add frame widget
  $AddFrame open $Connection
  # Open the search frame widget
  $SearchFrame open $Connection
  # Connect search dialogs.
  catch {$PrintLabels::PrintLabelsDialog open $Connection}
  catch {$Export::ExportCSVDialog open $Connection}
  catch {$Export::ExportEnvDialog open $Connection}
  catch {$Export::ExportLetterDialog open $Connection}
}


#*************************************
# Initialize a database
#*************************************
proc DirectMailTk::InitDatabase {} {
  variable Connection

  DatabaseWidgets::CreateAddressTable $Connection
}

#*************************************
# Hook function to start of search.
# Initialize search results list.
#*************************************
proc DirectMailTk::SearchBegin {} {
  variable Results
  $Results delete [$Results items]
  return true
}

#*************************************
# Process one row of results
# Insert item into search results list.
#*************************************
proc DirectMailTk::SearchRow {row} {
#  puts stderr "*** DirectMailTk::SearchRow $row"
  variable Results
   # Insert into results row
   $Results insert end [lindex $row 0] -text "[lindex $row 2] [lindex $row 3]" \
			-data $row
}

#*************************************
# Finish up search, enable the search buttons
#*************************************
proc DirectMailTk::SearchResults {count} {
  variable ResultButtons
  # And activate the results buttons.
  $ResultButtons configure -state normal
}


#*************************************
# Delete selected records
#*************************************
proc DirectMailTk::DeleteRecords {} {
  variable Connection
  variable Results

  # Get selection
  set selected [$Results selection get]
  if {[llength $selected] == 0} {return}

  # Verify the delete process.
  if {[string equal [tk_messageBox -type yesno -default no -icon question \
			-message "Are you sure you want to delete the selected records?"] no]} {
    return
  }
  # Delete records.
  foreach id $selected {
    $Results delete $id
    DatabaseWidgets::DeleteAddress $Connection $id
  }
}


#*************************************
# Add a page for a specified row
#*************************************
proc DirectMailTk::AddRecordPage {row {edit no}} {
  variable Connection
  variable MainNotebook

  # Get unique id
  set ident [lindex $row 0]
  # And Name
  set name  "[lindex $row 2] [lindex $row 3]"
  # Create the page
  set page [$MainNotebook insert end $ident  -text "$name ($ident)"]
  # Fill it in.
  RecordPage::RecordPage create $page -row $row -edit $edit \
				      -connection $Connection \
				      -notebook $MainNotebook \
				      -ident $ident
}

#*************************************
# Add editable records
#*************************************
proc DirectMailTk::EditRecords {} {
  variable Connection
  variable Results

  set selected [$Results selection get]
  if {[llength $selected] == 0} {return}

  foreach id $selected {
    AddRecordPage [$Results itemcget $id -data] yes
  }
}

#*************************************
# Add non editable records
#*************************************
proc DirectMailTk::ViewRecords {} {
  variable Connection
  variable Results

  set selected [$Results selection get]
  if {[llength $selected] == 0} {return}

  foreach id $selected {
    AddRecordPage [$Results itemcget $id -data]
  }
}

#*************************************
# Print Labels
#*************************************
proc DirectMailTk::PrintLabels::PrintDatabase {} {
  variable PrintLabelsDialog

  # Create print labels dialog, if needed
  if {[string equal "$PrintLabelsDialog" {}]} {
    set PrintLabelsDialog [::PrintLabels::PrintLabelsDialog .printLabels%AUTO%]
    $PrintLabelsDialog open $::DirectMailTk::Connection
  }
  # Print labels
  $PrintLabelsDialog draw -parent $::DirectMailTk::Main \
			  -title "Print selected rows"
}

#*************************************
# Export database to CSV
#*************************************
proc DirectMailTk::Export::ExportCSV {} {
  variable ExportCSVDialog

  # Create Export dialog
  if {[string equal "$ExportCSVDialog" {}]} {
    set ExportCSVDialog [DatabaseWidgets::ExportCSVDialog .exportCSV%AUTO%]
    $ExportCSVDialog open $::DirectMailTk::Connection
  }
  # Export rows
  $ExportCSVDialog draw -parent $::DirectMailTk::Main \
			-title "Export selected rows to CSV"
}

#*************************************
# Export database to Envelopes
#*************************************
proc DirectMailTk::Export::ExportENV {} {
  variable ExportEnvDialog

  # Create Export dialog
  if {[string equal "$ExportEnvDialog" {}]} {
    set ExportEnvDialog [DatabaseWidgets::ExportEnvDialog .exportEnv%AUTO%]
    $ExportEnvDialog open $::DirectMailTk::Connection
  }
  # Export rows
  $ExportEnvDialog draw -parent $::DirectMailTk::Main \
			-title "Export selected rows to Envlopes"
}


#*************************************
# Export database as letter environments
#*************************************
proc DirectMailTk::Export::ExportLetter {} {
  variable ExportLetterDialog

  # Create Export dialog
  if {[string equal "$ExportLetterDialog" {}]} {
    set ExportLetterDialog [DatabaseWidgets::ExportLetterDialog .exportLetter%AUTO%]
    $ExportLetterDialog open $::DirectMailTk::Connection
  }
  # Export rows
  $ExportLetterDialog draw -parent $::DirectMailTk::Main \
			-title "Export selected rows to Envlopes"
}


