#* 
#* ------------------------------------------------------------------
#* SelectDatasource.tcl - Data source selector
#* Created by Robert Heller on Mon Feb 26 13:34:11 2007
#* ------------------------------------------------------------------
#* Modification History: $Log: headerfile.text,v $
#* Modification History: Revision 1.1  2002/07/28 14:03:50  heller
#* Modification History: Add it copyright notice headers
#* Modification History:
#* ------------------------------------------------------------------
#* Contents:
#* ------------------------------------------------------------------
#*  
#*     Generic Project
#*     Copyright (C) 2005  Robert Heller D/B/A Deepwoods Software
#* 			51 Locke Hill Road
#* 			Wendell, MA 01379-9728
#* 
#*     This program is free software; you can redistribute it and/or modify
#*     it under the terms of the GNU General Public License as published by
#*     the Free Software Foundation; either version 2 of the License, or
#*     (at your option) any later version.
#* 
#*     This program is distributed in the hope that it will be useful,
#*     but WITHOUT ANY WARRANTY; without even the implied warranty of
#*     MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
#*     GNU General Public License for more details.
#* 
#*     You should have received a copy of the GNU General Public License
#*     along with this program; if not, write to the Free Software
#*     Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
#* 
#*  
#* 

#**********************************************************************
#*                                                                    *
#* Select Datasource dialog.  This file contains the code that        *
#* implements the Select Datasource Dialog box.  This dialog box is   *
#* used to select the data source connection string to connect to the *
#* database holding the address database.                             *
#*                                                                    *
#**********************************************************************

package require snit;#			Snit package
package require BWidget;#		BWidget package
package require snodbc;#                DB interface code (tclodbc compatible)

namespace eval SelectDatasource {
  #*****************************************
  # Snit ensemble type to implement a Datasource Dialog
  # Creates a data connection string from various data source options
  #*****************************************
  snit::type SelectDatasourceDialog {
    pragma -hastypedestroy no
    pragma -hasinstances no
    pragma -hastypeinfo no

    typecomponent dialog;#			Dialog box component
    typecomponent   driverlisttitle;#		Title frame for driver list
    typecomponent     driverlistsw;#		ScrollWindow for driver list
    typecomponent       driverlist;#		List of available drivers
    typecomponent   databasename;#		Name of the database
    typecomponent   servername;#		Name of the server
    typecomponent   username;#			Name of the user
    typecomponent   password;#			The User's password
    typecomponent   otheroptions;#		additional options
    typevariable _labelWidth 15;#		Standard label width

    #*****************************************
    # Type constructor -- built the dialog object
    #*****************************************
    typeconstructor {
      # Create the base dialog window.
      set dialog [Dialog::create .selectDatasourceDialog \
			-default 0 -cancel 1 -transient yes -modal local \
			-title "Select Datasource" -bitmap question]
      $dialog add -name connect -text Connect -command [mytypemethod _Connect]
      $dialog add -name cancel -text Cancel -command [mytypemethod _Cancel]
      $dialog add -name help -text Help -command "BWHelp::HelpTopic SelectDatasourceDialog"
      set frame [$dialog getframe]
      # Now for the dialog's widgets.
      #--------------------------------
      # Driver list
      set driverlisttitle [TitleFrame::create $frame.driverlisttf \
			-text {Available Drivers} -side center]
      pack $driverlisttitle -fill both -expand yes
      set driverlistsw [ScrolledWindow::create \
				[$frame.driverlisttf getframe].driverlistsw \
				-auto both -scrollbar both]
      pack $driverlistsw -fill both -expand yes
      set driverlist [ListBox::create [$driverlistsw getframe].driverlist \
			-selectfill yes -selectmode single]
      pack $driverlist -fill both -expand yes
      $driverlistsw setwidget $driverlist
      # Database name
      set databasename [LabelEntry::create $frame.databasename \
				-label {Database:} \
				-labelwidth $_labelWidth]
      pack $databasename -fill x
      # Servername
      set servername [LabelEntry::create $frame.servername \
				-label {Server Name:} \
				-labelwidth $_labelWidth \
				-text {localhost}]
      pack $servername -fill x
      # Username
      global tcl_platform
      set username [LabelEntry::create $frame.username \
				-label {Username:} \
				-labelwidth $_labelWidth \
				-text "$tcl_platform(user)"]
      pack $username -fill x
      # Password
      set password [LabelEntry::create $frame.password \
				-label {Password:} -show * \
				-labelwidth $_labelWidth]
      pack $password -fill x
      # Additional options
      set otheroptions [LabelEntry::create $frame.otheroptions \
				-label {Other Options:} \
				-labelwidth $_labelWidth]
      pack $otheroptions -fill x
      
    }
    #*****************************************
    # _Connect typemethod -- bound to Connect button
    # Check to see that at least a driver has been selected.
    #*****************************************
    typemethod _Connect {} {
      set selected [$driverlist selection get]
      if {[llength $selected] == 0} {
	tk_messageBox -type ok -icon error -message "Please select at least a driver!"
	return
      }
      $dialog withdraw
      return [$dialog enddialog connect]
    }
    #*****************************************
    # _Cancel typemethod -- bound to the Cancel button
    # Unconditionally end the dialog
    #*****************************************
    typemethod _Cancel {} {
      $dialog withdraw
      return [$dialog enddialog cancel]
    }
    #*****************************************
    # draw typemethod -- API entry point.  Initialize the driver list
    # and display the dialog box.  If Connect is selected, form a data
    # connection string from the specified values.
    #*****************************************
    typemethod draw {args} {
      set parent [from args -parent .];# Fetch the parent window
      $dialog configure -parent $parent;# Configure the parent in the dialog
      wm transient [winfo toplevel $dialog] $parent;# and set the transient
      # Rebuild the driver list
      $driverlist delete [$driverlist items]
      foreach driver [lsort -dictionary [database drivers]] {
	set key [lindex [split $driver] 0]
	$driverlist insert end \
		[string tolower [regsub -all {[[:space:]]} $key {}]] \
		-text "$driver" -data "$key"
      }
      # Display the dialog, and when the dialog is ended, switch on button
      switch [$dialog draw] {
	connect {;# Connect button -- form connection string
	  set selected [lindex [$driverlist selection get] 0]
	  set connectionString "Driver=[$driverlist itemcget $selected -data]"
	  set db "[$databasename cget -text]"
	  if {[string length "$db"] > 0} {append connectionString ";Database=$db"}
	  set sn "[$servername cget -text]"
	  if {[string length "$sn"] > 0} {append connectionString ";Servername=$sn"}
	  set un "[$username cget -text]"
	  if {[string length "$un"] > 0} {append connectionString ";Username=$un"}
	  set pw "[$password cget -text]"
	  if {[string length "$pw"] > 0} {append connectionString ";Password=$pw"}
	  set oo "[$otheroptions cget -text]"
	  if {[string length "$oo"] > 0} {append connectionString ";$oo"}
	  return "$connectionString"
        }
        cancel {;# Cancel button -- return the empty string
	  return {}
        }
      }
    }
  }
}

package provide SelectDatasource 1.0
