#* 
#* ------------------------------------------------------------------
#* PrintLabels.tcl - Label Printer
#* Created by Robert Heller on Sat Mar  3 10:48:00 2007
#* ------------------------------------------------------------------
#* Modification History: $Log: headerfile.text,v $
#* Modification History: Revision 1.1  2002/07/28 14:03:50  heller
#* Modification History: Add it copyright notice headers
#* Modification History:
#* ------------------------------------------------------------------
#* Contents:
#* ------------------------------------------------------------------
#*  
#*     Generic Project
#*     Copyright (C) 2005  Robert Heller D/B/A Deepwoods Software
#* 			51 Locke Hill Road
#* 			Wendell, MA 01379-9728
#* 
#*     This program is free software; you can redistribute it and/or modify
#*     it under the terms of the GNU General Public License as published by
#*     the Free Software Foundation; either version 2 of the License, or
#*     (at your option) any later version.
#* 
#*     This program is distributed in the hope that it will be useful,
#*     but WITHOUT ANY WARRANTY; without even the implied warranty of
#*     MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
#*     GNU General Public License for more details.
#* 
#*     You should have received a copy of the GNU General Public License
#*     along with this program; if not, write to the Free Software
#*     Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
#* 
#*  
#*


#**************************************************************************
#*                                                                        *
#* Code to print labels.  Portions of this code were 'lifted'/translated  *
#* from labelnation-1.164, (C) 2000-2005 Karl Fogel  <kfogel@red-bean.com>*
#*                                                                        *
#**************************************************************************

package require snit;#			Snit package
package require BWidget;#		BWidget package
package require DatabaseWidgets;#	Database Widgets package

namespace eval PrintLabels {
  # Directory where the supplied label parameter files live
  variable LabelParDir \
		[file join [file dirname \
				[file dirname \
					[file dirname \
						[info script]]]] \
			   LabelParameters]
  # File types for label parameter files
  variable LabelParFileTypes {
	{{Label Par Files} {.par} TEXT}
	{{All Files} * TEXT}
  }
#  puts stderr "*** LabelParDir = $LabelParDir"
  #*****************************************
  # Widget adaptor for printing labels (dialog box for editing label parameters)
  #*****************************************
  snit::widgetadaptor Label {
    # Components:
    component   leftmarginSB;#		Left Margin
    component   bottommarginSB;#	Bottom Margin
    component   labelwidthSB;#		Label Width
    component   labelheightSB;#		Label Height
    component   horizspaceSB;#		Horizontal Space
    component   vertspaceSB;#		Vertical Space
    component   horiznumlabelsSB;#	Horizontal Number of Labels
    component   vertnumlabelsSB;#	Vertical Number of Labels
    component   fontnameCB;#		Font Name
    component   fontsizeSB;#		Font Size
    # Options:
    delegate option -parent to hull;#	-parent delegated to the hull
    delegate option -title  to hull;#	-title  delegated to the hull
    # Name of the parameter file
    option -parameterfile -default {} -validatemethod _ValidFile \
				      -configuremethod _LoadParameterFile
    # Left Margin
    option -leftmargin -default 11 -validatemethod _ValidInteger
    # Bottom Margin
    option -bottommargin -default 62 -validatemethod _ValidInteger
    # Label Width
    option -labelwidth -default 300 -validatemethod _ValidInteger
    # Label Height
    option -labelheight -default 96 -validatemethod _ValidInteger
    # Horizontal space between labels
    option -horizspace -default 0 -validatemethod _ValidInteger
    # Vertical space between labels
    option -vertspace -default 0 -validatemethod _ValidInteger
    # Number of labels across the page
    option -horiznumlabels -default 2 -validatemethod _ValidInteger
    # Number of labels in each column on the page
    option -vertnumlabels -default 7 -validatemethod _ValidInteger
    # Name of the font to use
    option -fontname -default Times-Roman
    # Base size of the font
    option -fontsize -default 12 -validatemethod _ValidInteger
    # Option validation methods:
    # Check for a valid file (exists and readable)
    method _ValidFile {option value} {
      if {[file exists "$value"] && [file readable "$value"]} {
	return "$value"
      } else {
	error "Expected a existing, readable file for $option, got $value!"
      }
    }
   # Check for a valid positive integer
    method _ValidInteger {option value} {
      if {[string is integer -strict "$value"] && $value >= 0} {
	return "$value"
      } else {
	error "Expected a positive integer for $option, got $value!"
      }
    }
    # Instance variable:
    variable _PageNo 0;#		Current page number
    variable _X 0;#			Current X offset (in labels)
    variable _Y 0;#			Current Y offset (in labels)
    variable _StartNewPage no;#		Need to start a new page?
    variable _TextMargin;#		Text margin
    #*****************************************
    # Constructor -- create label parameter edit dialog
    #*****************************************
    constructor {args} {
      # Create the hull
      installhull using Dialog::create \
				-default 0 -cancel 1 -side bottom \
				-bitmap question -transient yes -modal local
#      $self _LoadFile -parameterfile "[from args -parameterfile]"
      $hull add -name ok     -text OK     -command [mymethod _OK]
      $hull add -name cancel -text Cancel -command [mymethod _Cancel]
      $hull add -name help   -text Help   \
				-command "BWHelp::HelpTopic CustomLabelDialog"
      set frame [$hull getframe]
      $self configurelist $args;#	Configure options
      # Build the rest of the GUI components
      set leftmarginLF [LabelFrame::create $frame.leftmarginLF \
				-text "LeftMargin:" -width $::DatabaseWidgets::LabelWidth]
      pack $leftmarginLF -fill x
      install leftmarginSB using SpinBox::create \
				[$leftmarginLF getframe].leftmarginSB \
				-range {0 612 1} -text $options(-leftmargin)
      pack $leftmarginSB -fill x -expand yes
      set bottommarginLF [LabelFrame::create $frame.bottommarginLF \
				-text "BottomMargin:" -width $::DatabaseWidgets::LabelWidth]
      pack $bottommarginLF -fill x
      install bottommarginSB using SpinBox::create \
				[$bottommarginLF getframe].bottommarginSB \
				-range {0 792 1} -text $options(-bottommargin)
      pack $bottommarginSB -fill x -expand yes
      set labelwidthLF [LabelFrame::create $frame.labelwidthLF \
				-text "LabelWidth:" -width $::DatabaseWidgets::LabelWidth]
      pack $labelwidthLF -fill x
      install labelwidthSB using SpinBox::create \
				[$labelwidthLF getframe].labelwidthSB \
				-range {1 612 1} -text $options(-labelwidth)
      pack $labelwidthSB -fill x -expand yes
      set labelheightLF [LabelFrame::create $frame.labelheightLF \
				-text "LabelHeight:" -width $::DatabaseWidgets::LabelWidth]
      pack $labelheightLF -fill x
      install labelheightSB using SpinBox::create \
				[$labelheightLF getframe].labelheightSB \
				-range {1 792 1} -text $options(-labelheight)
      pack $labelheightSB -fill x -expand yes
      set horizspaceLF [LabelFrame::create $frame.horizspaceLF \
				-text "HorizSpace:" -width $::DatabaseWidgets::LabelWidth]
      pack $horizspaceLF -fill x
      install horizspaceSB using SpinBox::create \
				[$horizspaceLF getframe].horizspaceSB \
				-range {0 612 1} -text $options(-horizspace)
      pack $horizspaceSB -fill x -expand yes
      set vertspaceLF [LabelFrame::create $frame.vertspaceLF \
				-text "VertSpace:" -width $::DatabaseWidgets::LabelWidth]
      pack $vertspaceLF -fill x
      install vertspaceSB using SpinBox::create \
				[$vertspaceLF getframe].vertspaceSB \
				-range {0 792 1} -text $options(-vertspace)
      pack $vertspaceSB -fill x -expand yes
      set horiznumlabelsLF [LabelFrame::create $frame.horiznumlabelsLF \
				-text "HorizNumLabels:" -width $::DatabaseWidgets::LabelWidth]
      pack $horiznumlabelsLF -fill x
      install horiznumlabelsSB using SpinBox::create \
				[$horiznumlabelsLF getframe].horiznumlabelsSB \
				-range {1 100 1} -text $options(-horiznumlabels)
      pack $horiznumlabelsSB -fill x -expand yes
      set vertnumlabelsLF [LabelFrame::create $frame.vertnumlabelsLF \
				-text "VertNumLabels:" -width $::DatabaseWidgets::LabelWidth]
      pack $vertnumlabelsLF -fill x
      install vertnumlabelsSB using SpinBox::create \
				[$vertnumlabelsLF getframe].vertnumlabelsSB \
				-range {1 50 1} -text $options(-vertnumlabels)
      pack $vertnumlabelsSB -fill x -expand yes
      set fontnameLF [LabelFrame::create $frame.fontnameLF \
				-text "fontname:" -width $::DatabaseWidgets::LabelWidth]
      pack $fontnameLF -fill x
      install fontnameCB using ComboBox::create \
				[$fontnameLF getframe].fontnameCB \
				-values {Courier Courier-Bold 
					 Courier-BoldOblique Courier-Oblique
					 Helvetica Helvetica-Bold
					 Helvetica-BoldOblique 
					 Helvetica-Oblique Symbol
					 Times-Bold Times-BoldItalic
					 Times-Italic Times-Roman
					 ZapfDingbats} \
				-text "$options(-fontname)"
      pack $fontnameCB -fill x -expand yes
      set fontsizeLF [LabelFrame::create $frame.fontsizeLF \
				-text "fontsize:" -width $::DatabaseWidgets::LabelWidth]
      pack $fontsizeLF -fill x
      install fontsizeSB using SpinBox::create \
				[$fontsizeLF getframe].fontsizeSB \
				-range {1 50 1} -text $options(-fontsize)
      pack $fontsizeSB -fill x -expand yes
    }
    #*****************************************
    # OK button clicked -- update label parameters from GUI elements
    #*****************************************
    method _OK {} {
      if {[catch {
	$self configure -leftmargin     "[$leftmarginSB cget -text]"
	$self configure -bottommargin   "[$bottommarginSB cget -text]"
	$self configure -labelwidth     "[$labelwidthSB cget -text]"
	$self configure -labelheight    "[$labelheightSB cget -text]"
	$self configure -horizspace     "[$horizspaceSB cget -text]"
	$self configure -vertspace      "[$vertspaceSB cget -text]"
	$self configure -horiznumlabels "[$horiznumlabelsSB cget -text]"
	$self configure -vertnumlabels  "[$vertnumlabelsSB cget -text]"
	$self configure -fontname       "[$fontnameCB cget -text]"
	$self configure -fontsize       "[$fontsizeSB cget -text]"
      } error]} {
        tk_messageBox -type ok -icon error -message "$error"
        return
      }
      $hull withdraw
      return [$hull enddialog ok]
    }
    #*****************************************
    # Cancel button clicked -- just dismis the dialog box
    #*****************************************
    method _Cancel {} {
      $hull withdraw
      return [$hull enddialog cancel]
    }
    #*****************************************
    # Start editing the label's parameters
    #*****************************************
    method editlabel {args} {
      $hull configure -parent [from args -parent]
      $hull configure -title  [from args -title]
      $leftmarginSB configure -text "[$self cget -leftmargin]"
      $bottommarginSB configure -text "[$self cget -bottommargin]"
      $labelwidthSB configure -text "[$self cget -labelwidth]"
      $labelheightSB configure -text "[$self cget -labelheight]"
      $horizspaceSB configure -text "[$self cget -horizspace]"
      $vertspaceSB configure -text "[$self cget -vertspace]"
      $horiznumlabelsSB configure -text "[$self cget -horiznumlabels]"
      $vertnumlabelsSB configure -text "[$self cget -vertnumlabels]"
      $fontnameCB configure -text "[$self cget -fontname]"
      $fontsizeSB configure -text "[$self cget -fontsize]"
      wm transient [winfo toplevel $win] [$hull cget -parent]
      return [$hull draw]
    }
    #*****************************************
    # Load a parameter file
    #*****************************************
    method _LoadParameterFile {option value} {
#      puts stderr "*** $self _LoadParameterFile $option $value"
      set options($option) "$value"
      if {[string equal "$value" {}]} {return}
      if {[catch {open "$value" r} pfp]} {
	error "Could not open $value: $pfp"
	return
      }
#      parray options
      while {[gets $pfp line] >= 0} {
	if {[regexp  {^([^:]*):[[:space:]]*(.*)$} "$line" -> key value] > 0} {
#	  puts stderr "*** $self _LoadParameterFile: key = $key, value = $value"
	  $self configure -[string tolower "$key"] "$value"
	}
      }
#      parray options
      close $pfp
    }
    #*****************************************
    # Quote Postscript strings
    #*****************************************
    method _quotePS {string} {
      regsub -all {[\\()]} "$string" {\\&} string
      return "$string"
    }
    #*****************************************
    # Set up for reencoding a font to use an ISO-8859 encoding
    #*****************************************
    method _SetupISO8859 {fp} {
      puts $fp {/deffont {
  findfont exch scalefont def
} bind def

/reencode_font {
  findfont reencode 2 copy definefont pop def
} bind def

% reencode the font
% <encoding-vector> <fontdict> -> <newfontdict>
/reencode { %def
  dup length 5 add dict begin
    { %forall
      1 index /FID ne
      { def }{ pop pop } ifelse
    } forall
    /Encoding exch def

    % Use the font's bounding box to determine the ascent, descent,
    % and overall height; don't forget that these values have to be
    % transformed using the font's matrix.
    % We use 'load' because sometimes BBox is executable, sometimes not.
    % Since we need 4 numbers an not an array avoid BBox from being executed
    /FontBBox load aload pop
    FontMatrix transform /Ascent exch def pop
    FontMatrix transform /Descent exch def pop
    /FontHeight Ascent Descent sub def

    % Define these in case they're not in the FontInfo (also, here
    % they're easier to get to.
    /UnderlinePosition 1 def
    /UnderlineThickness 1 def

    % Get the underline position and thickness if they're defined.
    currentdict /FontInfo known {
      FontInfo

      dup /UnderlinePosition known {
        dup /UnderlinePosition get
        0 exch FontMatrix transform exch pop
        /UnderlinePosition exch def
      } if

      dup /UnderlineThickness known {
        /UnderlineThickness get
        0 exch FontMatrix transform exch pop
        /UnderlineThickness exch def
      } if

    } if
    currentdict
  end
} bind def

/ISO-8859-1Encoding [
/.notdef /.notdef /.notdef /.notdef /.notdef /.notdef /.notdef /.notdef
/.notdef /.notdef /.notdef /.notdef /.notdef /.notdef /.notdef /.notdef
/.notdef /.notdef /.notdef /.notdef /.notdef /.notdef /.notdef /.notdef
/.notdef /.notdef /.notdef /.notdef /.notdef /.notdef /.notdef /.notdef
/space /exclam /quotedbl /numbersign /dollar /percent /ampersand /quoteright
/parenleft /parenright /asterisk /plus /comma /minus /period /slash
/zero /one /two /three /four /five /six /seven
/eight /nine /colon /semicolon /less /equal /greater /question
/at /A /B /C /D /E /F /G
/H /I /J /K /L /M /N /O
/P /Q /R /S /T /U /V /W
/X /Y /Z /bracketleft /backslash /bracketright /asciicircum /underscore
/quoteleft /a /b /c /d /e /f /g
/h /i /j /k /l /m /n /o
/p /q /r /s /t /u /v /w
/x /y /z /braceleft /bar /braceright /asciitilde /.notdef
/.notdef /.notdef /.notdef /.notdef /.notdef /.notdef /.notdef /.notdef
/.notdef /.notdef /.notdef /.notdef /.notdef /.notdef /.notdef /.notdef
/.notdef /.notdef /.notdef /.notdef /.notdef /.notdef /.notdef /.notdef
/.notdef /.notdef /.notdef /.notdef /.notdef /.notdef /.notdef /.notdef
/space /exclamdown /cent /sterling /currency /yen /brokenbar /section
/dieresis /copyright /ordfeminine /guillemotleft /logicalnot /hyphen /registered /macron
/degree /plusminus /twosuperior /threesuperior /acute /mu /paragraph /bullet
/cedilla /onesuperior /ordmasculine /guillemotright /onequarter /onehalf /threequarters /questiondown
/Agrave /Aacute /Acircumflex /Atilde /Adieresis /Aring /AE /Ccedilla
/Egrave /Eacute /Ecircumflex /Edieresis /Igrave /Iacute /Icircumflex /Idieresis
/Eth /Ntilde /Ograve /Oacute /Ocircumflex /Otilde /Odieresis /multiply
/Oslash /Ugrave /Uacute /Ucircumflex /Udieresis /Yacute /Thorn /germandbls
/agrave /aacute /acircumflex /atilde /adieresis /aring /ae /ccedilla
/egrave /eacute /ecircumflex /edieresis /igrave /iacute /icircumflex /idieresis
/eth /ntilde /ograve /oacute /ocircumflex /otilde /odieresis /divide
/oslash /ugrave /uacute /ucircumflex /udieresis /yacute /thorn /ydieresis
] def
}
    }
    #*****************************************
    # Create a Postscript clipping function
    #*****************************************
    method _MakeClippingFunction {fp innerMargin showBoundingBox} {
      set upperBound [expr {$options(-labelheight) - $innerMargin}]
      set rightBound [expr {$options(-labelwidth) - $innerMargin}]
      puts $fp "/labelclip \{"
      puts $fp "\tnewpath"
      puts $fp [format "\t%f %f moveto" $innerMargin $innerMargin]
      puts $fp [format "\t%f %f lineto" $rightBound  $innerMargin]
      puts $fp [format "\t%f %f lineto" $rightBound  $upperBound]
      puts $fp [format "\t%f %f lineto" $innerMargin $upperBound]
      puts $fp "\tclosepath"
      puts $fp "\tclip"
      if {$showBoundingBox} {puts $fp "\tstroke"}
      puts $fp "\} def"
    }
    #*****************************************
    # Output a Postscript preamble and initialize for printing labels
    #*****************************************
    method preamble {fp {showBoundingBox no} {innerMargin 1}} {
      puts $fp {%!PS-Adobe-3.0}
      puts $fp {%%Pages: (atend)}
      puts $fp {%%Title: Labels}
      puts $fp {%%Creator: DirectMailTk Version 0.0}
      puts $fp "%%CreationDate: [clock format [clock scan now]]"
      set bbox "%%BoundingBox: $options(-leftmargin) $options(-bottommargin) "
      set right [expr {$options(-leftmargin) + 
		       ($options(-labelwidth) * $options(-horiznumlabels)) + 
		       ($options(-horizspace) * ($options(-horiznumlabels)-1))}]
      append bbox "$right "
      set top [expr {$options(-bottommargin) +
		     ($options(-labelheight) * $options(-vertnumlabels)) +
		     ($options(-vertspace) * ($options(-vertnumlabels)-1))}]
      append bbox "$top"
      puts $fp "$bbox"
      puts $fp {%%EndComments}
      $self _SetupISO8859 $fp
      puts $fp [format {/ISO%s ISO-8859-1Encoding /%s reencode_font} \
			"$options(-fontname)" "$options(-fontname)"]
      $self _MakeClippingFunction $fp $innerMargin $showBoundingBox
      puts $fp {%%EndProlog}
      set _PageNo 0
      set _StartNewPage yes
      set _X 0
      set _Y 0
      set _TextMargin [expr {$innerMargin + 2.0}]
    }
    #*****************************************
    # Print one label
    #*****************************************
    method dolabel {lines fp} {
      set codeAccum {}
      set numLines [llength $lines]
      set upMostLineStart [expr {$options(-labelheight) / ($numLines + 1.0) * \
				 double($numLines)}]
      set distanceDown [expr {$options(-labelheight) / ($numLines + 2.0)}]
      set fontsize [expr {$options(-fontsize) / 
			  (1.0 + (($numLines - 4.0) / 10.0))}]
      append codeAccum "newpath\n"
      append codeAccum [format "ISO%s %f scalefont setfont\n" \
				"$options(-fontname)" $fontsize]
      set labelLine -1
      foreach line $lines {
	incr labelLine
	set thisLine [expr {$upMostLineStart - ($labelLine * $distanceDown)}]
	append codeAccum [format "%f %f moveto\n" $_TextMargin $thisLine]
	append codeAccum "([$self _quotePS $line]) show\n"
      }
      append codeAccum "stroke\n"
      if {$_StartNewPage} {
	incr _PageNo
	puts $fp [format {%%%%Page: %d} $_PageNo]
	puts $fp {%%BeginPageSetup}
	puts $fp [format {%f %f translate} \
			 $options(-leftmargin) $options(-bottommargin)]
	puts $fp {%%EndPageSetup}
	set _StartNewPage no
      }
      set thisXStep [expr {$_X * ($options(-labelwidth) + 
				  $options(-horizspace))}]
      set thisYStep [expr {$_Y * ($options(-labelheight) + 
				  $options(-vertspace))}]
      puts $fp {gsave}
      puts $fp [format "%f %f translate\n" $thisXStep $thisYStep]
      puts $fp {labelclip}
      puts $fp "$codeAccum"
      puts $fp {grestore}
      puts $fp {}
      incr _Y
      if {$_Y >= $options(-vertnumlabels)} {
        set _Y 0
	incr _X
      }
      if {$_X >= $options(-horiznumlabels)} {
	set _X 0
	puts $fp {showpage}
	set _StartNewPage yes
      }
    }
    #*****************************************
    # Print Postscript trailer
    #*****************************************
    method endoffile {fp} {
      if {$_Y > 0 || $_X > 0} {
	puts $fp {showpage}
      }
      puts $fp [format "%%Pages: %d" $_PageNo]
      puts $fp {%%EOF}
    }
  }
  #*****************************************
  # Print Labels Dialog widget adaptor
  #*****************************************
  snit::widgetadaptor PrintLabelsDialog {
    #*****************************************
    # Additional components
    component labelSpecDialog;#		Label spec object
    component labelfileLF;#		Label file LF
    component   labelfileE;#		Label file entry
    component   labelfileLoad;#		Label file load button
    component   labelfileBrowse;#	Label file browse button
    component editLabelB;#		Edit Label button
    component printSelectTF;#		Printer selection title frame
    component   printerCB;#		Printer selection
    component   fileE;#			File entry
    component   fileBrowse;#		File Browse button
    # Postscript file types.
    typevariable _PSFileTypes {
	{{Postscript files} {.ps} TEXT}
	{{All Files} * TEXT}
    }
    # Type variables
    typevariable _shortLabelWidth 10;#	Short label width
    # Instance variables
    variable _printerOrFile printer;#	Printer or file
    variable _FP;#			Open output file / pipe
    #*****************************************
    # Constructor.  After building the standard search frame dialog, include
    # some additional components
    #*****************************************
    DatabaseWidgets::SearchDialogConstructor PrintLabelsDialog Print {
      # Additional components:
      # Label Specification dialog
      install labelSpecDialog using PrintLabels::Label $win.labelSpecDialog
      # Label file name, browsing, loading, and editing
      install labelfileLF using LabelFrame::create \
			[$hull getframe].labelfileLF \
			-text "Label Spec. file:" \
			-width $::DatabaseWidgets::LabelWidth
      pack $labelfileLF -fill x
      install labelfileE using Entry::create \
			[$labelfileLF getframe].labelfileE
      pack $labelfileE -fill x -expand yes -side left
      install labelfileLoad using Button::create \
      			[$labelfileLF getframe].labelfileLoad \
			-text Load -command [mymethod _LabelFileLoad]
      pack $labelfileLoad -side left
      install labelfileBrowse using Button::create \
      			[$labelfileLF getframe].labelfileBrowse \
			-text Browse -command [mymethod _LabelFileBrowse]
      pack $labelfileBrowse -side right
      install editLabelB using Button::create [$hull getframe].editLabelB \
			-text "Edit Label Spec." \
			-command [mymethod _EditLabelSpec]
      pack $editLabelB -fill x
      # Print output selection, file or printer, which printer, what file
      install printSelectTF using \
			TitleFrame::create [$hull getframe].printSelectTF \
			-text "Printer Or File" -side center
      pack $printSelectTF -fill both -expand yes
      set prFrame [$printSelectTF getframe]
      grid [radiobutton $prFrame.prrb -text {} -value printer \
				-variable [myvar _printerOrFile] \
				-command [mymethod _TogglePrinterFile]] \
	-row 0 -column 0 -sticky nws
      grid [radiobutton $prFrame.firb -text {} -value file \
				-variable [myvar _printerOrFile] \
				-command [mymethod _TogglePrinterFile]] \
	-row 1 -column 0 -sticky nws
      grid columnconfigure $prFrame 0 -weight 0
      set prlf [LabelFrame::create $prFrame.prlf -text "Printer:" \
						 -width $_shortLabelWidth]
      grid $prlf -row 0 -column 1 -sticky news
      set filf [LabelFrame::create $prFrame.filf -text "File:" \
						 -width $_shortLabelWidth]
      grid $filf -row 1 -column 1 -sticky news
      grid columnconfigure $prFrame 1 -weight 1
      install printerCB using ComboBox::create [$prlf getframe].printerCB -editable no
      pack $printerCB -expand yes -fill x -side left
      install fileE using Entry::create [$filf getframe].fileE -state disabled
      pack $fileE -expand yes -fill x -side left
      install fileBrowse using Button::create [$filf getframe].fileBrowse \
					-text Browse -state disabled \
					-command [mymethod _BrowsePSFile]
      pack $fileBrowse -side right
    }
    #*****************************************
    # Toggle state for printer or file
    #*****************************************
    method _TogglePrinterFile {} {
      switch $_printerOrFile {
	printer {
	  $printerCB configure -state normal
	  $fileE configure -state disabled
	  $fileBrowse configure -state disabled
	}
	file {
	  $printerCB configure -state disabled
	  $fileE configure -state normal
	  $fileBrowse configure -state normal
	}
      }
    }
    #*****************************************
    # Browse for a Postscript file
    #*****************************************
    method _BrowsePSFile {} {
      set newfile [tk_getSaveFile -title "Postscript file to print to" \
				  -parent $win \
				  -filetypes $_PSFileTypes \
				  -initialfile "[$fileE cget -text]" \
				  -defaultextension .ps]
      if {[string length "$newfile"] > 0} {
	$fileE configure -text "$newfile"
      }
    }
    #*****************************************
    # Edit the label specification
    #*****************************************
    method _EditLabelSpec {} {
      $labelSpecDialog editlabel -parent $win
    }
    #*****************************************
    # Browse for a label specification file
    #*****************************************
    method _LabelFileBrowse {} {
      set newfile [tk_getOpenFile -title "Label File" -parent $win \
			-initialfile "[$labelfileE cget -text]" \
			-initialdir  $::PrintLabels::LabelParDir \
			-filetypes   $::PrintLabels::LabelParFileTypes]
      if {[string length "$newfile"] > 0} {
	$labelfileE configure -text "$newfile"
      }
    }
    #*****************************************
    # Load a label specification file
    #*****************************************
    method _LabelFileLoad {} {
      set file "[$labelfileE cget -text]"
      if {[string length "$file"] > 0} {
	$labelSpecDialog configure -parameterfile "$file"
      }
    }
    #*****************************************
    # Type method to find all printers
    #*****************************************
    typemethod _AllPrinters {} {
      global tcl_platform
      switch $tcl_platform(platform) {
	windows {return {}}
	macintosh -
	unix {
	  set lpstat [auto_execok lpstat]
	  if {[string equal "$lpstat" {}]} {return {}}
	  set printers {}
	  if {[catch {open "|$lpstat -a" r} lpstatFP]} {return {}}
	  while {[gets $lpstatFP line] >= 0} {
	    if {[regexp {^(.+)[[:space:]]accepting} "$line" -> printer] > 0} {
	      lappend printers $printer
	    }
	  }
	  close $lpstatFP
	  return $printers
	}
      }
    }
    #*****************************************
    # Type method to find the default printer
    #*****************************************
    typemethod _DefaultPrinter {} {
      global tcl_platform
      switch $tcl_platform(platform) {
	windows {return {}}
	macintosh -
	unix {
	  set lpstat [auto_execok lpstat]
	  if {[string equal "$lpstat" {}]} {return {}}
	  if {[catch {open "|$lpstat -d" r} lpstatFP]} {return {}}
          set defprinter {}
	  while {[gets $lpstatFP line] >= 0} {
	    if {[regexp {destination:[[:space:]](.*)$} "$line" -> defprinter] > 0} {
	      break
	    }
	  }
	  close $lpstatFP
	  return $defprinter
	}
      }
    }
    #*****************************************
    # Draw hook: set up printers: all and default
    #*****************************************
    method _DrawHook {} {
      # Get a fresh list of all printers.
      set allprinters [$type _AllPrinters]
#      puts stderr "*** $type draw: allprinters = $allprinters"
      # Configure the printer combo box
      $printerCB configure -values $allprinters
      # Fetch and configure for the default printer
      set defaultInd [lsearch $allprinters [$type _DefaultPrinter]]
#      puts stderr "*** $type draw: defaultInd = $defaultInd"
      $printerCB setvalue @$defaultInd
    }
    #*****************************************
    # Method to open a writable channel to the printer output (output file or
    # pipe to printer command
    #*****************************************
    method _OpenPSPrinterPipe {} {
      switch $_printerOrFile {
	printer {
	  global tcl_platform
	  switch $tcl_platform(platform) {
	    windows {return {}}
	    macintosh -
	    unix {
	      set printer "[$printerCB cget -text]"
	      set lp [auto_execok lp]
	      set lpr [auto_execok lpr]
	      if {![string equal "$lp" {}]} {
		set cmd "|$lp -d $printer"
	      } elseif {![string equal "$lpr" {}]} {
		set cmd "|$lpr -P$printer"
	      } else {
		tk_messageBox -type ok -icon error -message "Could not find print command -- neither lp nor lpr found!
		return {}
	      }
	      if {[catch {open "$cmd" w} pipeFP]} {
		tk_messageBox -type ok -icon error -message "Could not open $cmd: $pipeFP"
		return {}
	      } else {
		return $pipeFP
	      }
	    }
	  }
	}
	file {
	  set file "[$fileE cget -text]"
	  if {[catch {open "$file" w} pipeFP]} {
	    tk_messageBox -type ok -icon error -message "Could not open $file: $pipeFP"
	    return {}
	  } else {
	    return $pipeFP
	  }
	}
      }
    }
    #*****************************************
    # Printer start hook: open printer pipe and write the Postscript preamble
    #*****************************************
    method _StartHook {} {
      set _FP [$self _OpenPSPrinterPipe]
      $labelSpecDialog preamble $_FP;# showBoundingBox innerMargin
    }
    #*****************************************
    # Print one label.  Get row data and pass it onto the label printing function
    #*****************************************
    method _RowFunction {row} {
      set lines {}
      set line {}
      set sal [lindex $row 1]
      set fn  [lindex $row 2]
      set ln  [lindex $row 3]
      set suf [string trim [lindex $row 4]]
      if {[string length "$sal"] > 0} {append line "$sal "}
      append line "$fn $ln"
      if {[string length "$suf"] > 0} {append line ", $suf"}
      #	Name line, built from salutation, first name, last name, and suffix
      lappend lines "$line"
      # Organization line (only if Organization is not empty)
      set org [lindex $row 5]
      if {[string length "$org"] > 0} {lappend lines "$org"}
      lappend lines "[lindex $row 6]";#	Address line 1 line
      # Address2 line (only if not empty)
      set addr2 [lindex $row 7]
      if {[string length "$addr2"] > 0} {lappend lines "$addr2"}
      # City, State Zip line
      lappend lines "[lindex $row 8], [lindex $row 9] [lindex $row 10]"
      $labelSpecDialog dolabel $lines $_FP
    }
    #*****************************************
    # Result hook: print Postscript trailer and close the file/pipe
    #*****************************************
    method _ResultHook {count} {
      $labelSpecDialog endoffile $_FP
      close $_FP
      set _FP {}
      tk_messageBox -type ok -icon info -message "$count labels printed"
    }
  }
}


package provide PrintLabels 1.0
