#* 
#* ------------------------------------------------------------------
#* DatabaseWidgets.tcl - Database widgets
#* Created by Robert Heller on Fri Mar  2 10:01:26 2007
#* ------------------------------------------------------------------
#* Modification History: $Log: headerfile.text,v $
#* Modification History: Revision 1.1  2002/07/28 14:03:50  heller
#* Modification History: Add it copyright notice headers
#* Modification History:
#* ------------------------------------------------------------------
#* Contents:
#* ------------------------------------------------------------------
#*  
#*     Generic Project
#*     Copyright (C) 2005  Robert Heller D/B/A Deepwoods Software
#* 			51 Locke Hill Road
#* 			Wendell, MA 01379-9728
#* 
#*     This program is free software; you can redistribute it and/or modify
#*     it under the terms of the GNU General Public License as published by
#*     the Free Software Foundation; either version 2 of the License, or
#*     (at your option) any later version.
#* 
#*     This program is distributed in the hope that it will be useful,
#*     but WITHOUT ANY WARRANTY; without even the implied warranty of
#*     MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
#*     GNU General Public License for more details.
#* 
#*     You should have received a copy of the GNU General Public License
#*     along with this program; if not, write to the Free Software
#*     Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
#* 
#*  
#* 

#**********************************************************************
#*                                                                    *
#* Database Widgets.  Various widgets and things that interface to the*
#* database.
#*                                                                    *
#**********************************************************************

package require snit;#		Snit package
package require BWidget;#	BWidget package
package require snodbc;#	snodbc (database) package

namespace eval DatabaseWidgets {
  # Constant information about the address database
  variable TableName addresses;#	Name of the table
  # Table specification
  variable TableSpecification {(id integer not null unique,
	 salutation char(5),
	 firstname text not null, 
	 lastname text not null, 
	 suffix char(10),
	 organization text,
	 address1 text not null,
	 address2 text,
	 city text not null,
	 state char(2) not null,
	 zip char(10) not null,
	 telephone char(15) not null,
	 fax char(15))}
  # List of standard Salutations
  variable Salutations {{} Mr. Miss. Mrs. Ms. Dr. Prof.};# Salutations
  # Dummy type to generate unique ids
  snit::type statementID {
    pragma -hasinstances false
    typevariable id 0
    typemethod create {} {
      incr id
      return "${type}${id}"
    }
  }
  # All state abbrivs (from USPS.com)
  variable AllStates {AL AK AS AZ AR CA CO CT DE DC FM FL GA GU HI ID IL IN
		      IA KS KY LA ME MH MD MA MI MN MS MO MT NE NV NH NJ NM
		      NY NC ND MP OH OK OR PW PA PR RI SC SD TN TX UT VT VI
		      VA WA WV WI WY}
  # Version for searches (includes the null string)
  variable AllStatesSearch [concat {{}} $AllStates]
  variable LabelWidth 15;# Standard label width
  #**********************************
  # Add record frame
  #**********************************
  snit::widget AddFrame {
    variable _Connection {};#		Database connection object
    variable _InsertAddress {};#	Insert statement object
    variable _MaxIndex {};#		Max Index statement object

    # Frame components
    component nameLF;#			Name
    component   salutationCB;#		Salutation
    component   firstnameE;#		First Name
    component   lastnameE;#		Last Name
    component   suffixE;#		Suffix
    component organizationLE;#		Organization
    component address1LE;#		Address1
    component address2LE;#		Address2
    component cityStateZipLF;#		City, State, Zip frame
    component cityE;#			City
    component stateCB;#			State	
    component zipE;#			Zip
    component phoneLE;#			Phone
    component faxLE;#			Fax
    component button;#			Add Record Button

    #***************************************
    # Constructor: populate frame with GUI elememnts.
    #***************************************
    constructor {args} {
      # Name label frame.  Contains the salutation, firstname, lastname, and suffix
      install nameLF using LabelFrame::create $win.nameLF \
				-text "Name:" \
				-width $::DatabaseWidgets::LabelWidth \
				-state disabled
      pack $nameLF -fill x
      set nameLFfr [$nameLF getframe]
      # Salutation combo box
      install salutationCB using ComboBox::create $nameLFfr.salutationCB \
				-values $::DatabaseWidgets::Salutations \
				-editable no -state disabled
      $salutationCB setvalue first
      pack $salutationCB -side left
      # First name entry
      install firstnameE using Entry::create $nameLFfr.firstnameE \
				-state disabled
      pack $firstnameE -fill x -expand yes -side left
      # Last name entry
      install lastnameE using Entry::create $nameLFfr.lastnameE \
				-state disabled
      pack $lastnameE -fill x -expand yes -side left
      # Suffix entry
      install suffixE using Entry::create $nameLFfr.suffixE \
				-state disabled -width 10
      pack $suffixE -fill x -side right
      # Organization label entry
      install organizationLE using LabelEntry::create $win.organizationLE \
				-label "Organization:" \
				-labelwidth $::DatabaseWidgets::LabelWidth \
				-state disabled
      pack $organizationLE -fill x
      # Address line 1 label entry
      install address1LE using LabelEntry::create $win.address1LE \
				-label "Address 1:" \
				-labelwidth $::DatabaseWidgets::LabelWidth \
				-state disabled
      pack $address1LE -fill x
      # Address line 2 label entry
      install address2LE using LabelEntry::create $win.address2LE \
				-label "Address 2:" \
				-labelwidth $::DatabaseWidgets::LabelWidth \
				-state disabled
      pack $address2LE -fill x
      # City, State, and Zipcode label frame
      install cityStateZipLF using LabelFrame::create $win.cityStateZipLF \
				-text "City, State, Zip:" \
				-width $::DatabaseWidgets::LabelWidth \
				-state disabled
      pack $cityStateZipLF -fill x
      set cszfr [$cityStateZipLF getframe]
      # City entry
      install cityE using Entry::create $cszfr.cityE -state disabled
      pack $cityE -fill x -expand yes -side left
      # State combo box
      install stateCB using ComboBox::create $cszfr.stateCB -editable no \
					-values $::DatabaseWidgets::AllStates \
					-width 3 -state disabled
      $stateCB setvalue first
      pack $stateCB -side left
      # Zip entry
      install zipE using Entry::create $cszfr.zipE -width 10 -state disabled
      pack $zipE -side left
      # Telephone label entry
      install phoneLE using LabelEntry::create $win.phoneLE \
				-label "Telephone:" \
				-labelwidth $::DatabaseWidgets::LabelWidth \
				-state disabled
      pack $phoneLE -fill x
      # Fax label entry
      install faxLE using LabelEntry::create $win.faxLE \
				-label "Fax:" \
				-labelwidth $::DatabaseWidgets::LabelWidth \
				-state disabled
      pack $faxLE -fill x
      # Add entry button
      install button using Button::create $win.button \
				-text "Add Record" \
				-command [mymethod _AddRecord] \
				-state disabled
      pack $button -fill x
      #$self configurelist $args
    }
    #***************************************
    # Method open: connect to a database. Create statement objects to 
    # add records and enable all GUI elements.
    #***************************************
    method open {connection} {
      catch {$self close};#		Close existing connection, if any
      set _Connection $connection;#	Save current connection
      # Create statement objects
      set _InsertAddress \
		[$_Connection \
			statement [DatabaseWidgets::statementID create] \
			{insert into addresses values (?,?,?,?,?,?,?,?,?,?,?,?,?)} \
			{INTEGER CHAR CHAR CHAR CHAR CHAR CHAR CHAR CHAR CHAR CHAR CHAR CHAR}]
      set _MaxIndex \
		[$_Connection \
			statement [DatabaseWidgets::statementID create] \
			{select max(id) from addresses}]
      # Enable all of the GUI elememnts
      $nameLF configure -state normal
      $salutationCB configure -state normal
      $firstnameE configure -state normal
      $lastnameE configure -state normal
      $suffixE configure -state normal
      $organizationLE configure -state normal
      $address1LE configure -state normal
      $address2LE configure -state normal
      $cityStateZipLF configure -state normal
      $cityE configure -state normal
      $stateCB configure -state normal
      $zipE configure -state normal
      $phoneLE configure -state normal
      $faxLE configure -state normal
      $button configure -state normal
    }
    #***************************************
    # Method close: drop statement object and disable all GUI elememnts.
    #***************************************
    method close {} {
      # Drop statement object
      catch {$_InsertAddress drop}
      set _InsertAddress {}
      catch {$_MaxIndex drop}
      set _MaxIndex {}
      # Reset connect
      set _Connection {}
      # Disable all GUI elememnts
      $nameLF configure -state disabled
      $salutationCB configure -state disabled
      $firstnameE configure -state disabled
      $lastnameE configure -state disabled
      $suffixE configure -state disabled
      $organizationLE configure -state disabled
      $address1LE configure -state disabled
      $address2LE configure -state disabled
      $cityStateZipLF configure -state disabled
      $cityE configure -state disabled
      $stateCB configure -state disabled
      $zipE configure -state disabled
      $phoneLE configure -state disabled
      $faxLE configure -state disabled
      $button configure -state disabled
    }
    #***************************************
    # Method _AddRecord: insert the contents of the GUI elements into the 
    # database.
    #***************************************
    method _AddRecord {} {
      # Make sure we have a table to add rows to.
      if {[catch {$_Connection tables addresses} tableinfo]} {
	puts stderr "*** $self _AddRecord: tableinfo = $tableinfo"
        tk_messageBox -type ok -icon error -message "Please initialize the database first!"
	return
      }
      # Extract fields from the GUI and then insert the row.
      #-------
      # Get a fresh ID number
      set maxindex [$_MaxIndex run]
      set newindex [lindex $maxindex 0]
      if {[string equal "$newindex" {{}}]} {set newindex 0}
      incr newindex
      # Insert into database
      set res [catch {$_InsertAddress run [list \
		$newindex \
		"[$salutationCB cget -text]" \
		"[$firstnameE cget -text]" \
		"[$lastnameE cget -text]" \
		"[$suffixE cget -text]" \
		"[$organizationLE cget -text]" \
		"[$address1LE cget -text]" \
		"[$address2LE cget -text]" \
		"[$cityE cget -text]" \
		"[$stateCB cget -text]" \
		"[$zipE cget -text]" \
		"[$phoneLE cget -text]" \
		"[$faxLE cget -text]"]} error]
      if {$res} {tk_messageBox -type ok -icon error -message "$error"}
    }
    #***************************************
    # Destructor: close database connection, if any.
    #***************************************
    destructor {
      catch {$self close}
    }
  }
  #***************************************
  # General search frame
  #***************************************
  snit::widget SearchFrame {
    variable _Connection {};#		Database connection object
    variable _searchStatement {};#	Current search statement
    variable _Agregator AND;#		Agregrator

    # Search frame components
    component leftTF;#			Left (search terms) title frame
    component agregatorLF;#		Agregrator label frame
    component   andRB;#			And Radiobutton
    component   orRB;#			Or Radiobutton
    component firstnameLF;#		First Name label frame
    component   firstnameCompCB;#	Compariator ComboBox
    component   firstnameStringE;#	String to compare to
    component lastnameLF;#		Last Name label frame
    component   lastnameCompCB;#	Compariator ComboBox
    component   lastnameStringE;#	String to compare to
    component organizationLF;#		Organization label frame
    component   organizationCompCB;#	Compariator ComboBox
    component   organizationStringE;#	String to compare to
    component address1LF;#		Address1 label frame
    component   address1CompCB;#	Compariator ComboBox
    component   address1StringE;#	String to compare to
    component address2LF;#		Address2 label frame
    component   address2CompCB;#	Compariator ComboBox
    component   address2StringE;#	String to compare to
    component cityLF;#			City label frame
    component   cityCompCB;#		Compariator ComboBox
    component   cityStringE;#		String to compare to
    component stateLF;#			State label frame
    component   stateCompCB;#		Compariator ComboBox
    component   stateCB;#		String to compare to
    component zipLF;#			Zip label frame
    component   zipCompCB;#		Compariator ComboBox
    component   zipStringE;#		String to compare to
    component phoneLF;#			Phone label frame
    component   phoneCompCB;#		Compariator ComboBox
    component   phoneStringE;#		String to compare to
    component faxLF;#			Fax label frame
    component   faxCompCB;#		Compariator ComboBox
    component   faxStringE;#		String to compare to
    component searchButtons;#		Search buttons
    component rightTF;#			Right (where clause) title frame
    component clauseList;#		Where clause list
    component clauseButtons;#		Where clause buttons

    # Options:
    option -rowfunction;#		Row function.
    option -searchresulthook;#		Search Results hook script
    option -searchstarthook;#		Search Start hook script
    option -callbackargs -default {};#	Callback args
    option -includesearch -default yes;# Include search button?
    #****************************************
    #* Helper method to create a entry-style line item
    #****************************************
    method _searchElementLFE {parent what label} {
      # Label frame
      install ${what}LF using LabelFrame::create $parent.${what}LF \
				-text "$label" \
				-width $::DatabaseWidgets::LabelWidth \
				-state disabled
      pack [set ${what}LF] -fill x
      set ${what}fr [[set ${what}LF] getframe]
      # Comparitor combo box
      install ${what}CompCB using ComboBox::create [set ${what}fr].${what}CompCB \
				-editable no \
				-values {contains {does not contain}} \
				-state disabled
      [set ${what}CompCB] setvalue first
      pack [set ${what}CompCB] -side left
      # Compare string entry
      install ${what}StringE using Entry::create [set ${what}fr].${what}StringE \
					-state disabled
      pack [set ${what}StringE] -side left -fill x -expand yes
    }
    #***************************************
    # Constructor: populate GUI elements
    #***************************************
    constructor {args} {
      $self configurelist $args;#		Get our options
      # Left side title frame: clause building GUI elements
      install leftTF using TitleFrame::create $win.leftTF \
				-text "Search Terms" \
				-side center -state disabled
      pack $leftTF -side left -expand yes -fill both
      set left [$leftTF getframe]
      # Clause agregator RadioButtons
      install agregatorLF using LabelFrame::create $left.agregatorLF \
				-text "Agregator:" \
				-width $::DatabaseWidgets::LabelWidth \
				-state disabled
      pack $agregatorLF -fill x
      set agregatorfr [$agregatorLF getframe]
      install andRB using radiobutton $agregatorfr.andRB -text AND -value AND \
				      -variable [myvar _Agregator] \
				      -state disabled
      pack $andRB -fill x -side left -expand yes
      install orRB using radiobutton $agregatorfr.orRB -text OR -value OR \
				      -variable [myvar _Agregator] \
				      -state disabled
      pack $orRB -fill x -side left -expand yes
      # Search elements (clause items)
      $self _searchElementLFE $left firstname {First Name:}
      $self _searchElementLFE $left lastname {Last Name:}
      $self _searchElementLFE $left organization Organization:
      $self _searchElementLFE $left address1 {Address 1:}
      $self _searchElementLFE $left address2 {Address 2:}
      $self _searchElementLFE $left city City:
      # State is different -- has a different compariator and uses a combo box
      # instead of an entry.
      install stateLF using LabelFrame::create $left.stateLF \
				-text "State:" \
				-width $::DatabaseWidgets::LabelWidth \
				-state disabled
      pack $stateLF -fill x
      set statefr [$stateLF getframe]
      install stateCompCB using ComboBox::create $statefr.stateCompCB \
				-editable no \
				-values {is {is not}} -state disabled
      $stateCompCB setvalue first
      pack $stateCompCB -side left
      install stateCB using ComboBox::create $statefr.stateCB \
				-editable no \
				-values $::DatabaseWidgets::AllStatesSearch \
				-state disabled
      pack $stateCB -side left -fill x -expand yes
      $stateCB setvalue first
      $self _searchElementLFE $left zip Zip:
      $self _searchElementLFE $left phone Telephone:
      $self _searchElementLFE $left fax Fax:
      # Search buttons (left side)
      install searchButtons using ButtonBox::create $left.searchButtons \
				-orient horizontal -homogeneous yes\
				-state disabled
      pack $searchButtons -fill x -side bottom
      # Dialog boxes have their Search button at the bottom
      if {$options(-includesearch)} {
        $searchButtons add -name search -text "Search" \
				-command [mymethod dosearch] \
				-state disabled
      }
      # Just add the clause
      $searchButtons add -name addclause -text "Add Clause" \
				-command [mymethod _AddClause] \
				-state disabled
      install rightTF using TitleFrame::create $win.rightTF \
				-text "Where clause" \
				-side center -state disabled
      # Right side title frame: list of clauses
      pack $rightTF -side right -expand yes -fill both
      set right [$rightTF getframe]
      set clauseListSW [ScrolledWindow::create $right.clauseListSW \
						-scrollbar both -auto both]
      pack $clauseListSW -expand yes -fill both
      install clauseList using ListBox::create \
				[$clauseListSW getframe].clauseList \
				-selectfill yes -selectmode single \
				-state disabled
      pack $clauseList -expand yes -fill both
      $clauseListSW setwidget $clauseList
      # Clause buttons: delete one or all clauses
      install clauseButtons using ButtonBox::create $right.clauseButtons \
				-orient horizontal -homogeneous yes\
				-state disabled
      pack $clauseButtons -fill x -side bottom
      $clauseButtons add -name deleteClause -text "Delete Clause" \
      				-command [mymethod _DeleteClause] \
				-state disabled
      $clauseButtons add -name newSearch -text "New Search" \
      				-command [mymethod _NewSearch] \
				-state disabled
    }
    #***************************************
    # open method -- open things up.  Get connected to the database and
    # enable all of the GUI elements
    #***************************************
    method open {connection} {
      catch {$self close};#             Close existing connection, if any
      set _Connection $connection
      $leftTF configure -state normal
      $agregatorLF configure -state normal
      $andRB configure -state normal
      $orRB  configure -state normal
      $firstnameLF configure -state normal
      $firstnameCompCB configure -state normal
      $firstnameStringE configure -state normal
      $lastnameLF configure -state normal
      $lastnameCompCB configure -state normal
      $lastnameStringE configure -state normal
      $organizationLF configure -state normal
      $organizationLF configure -state normal
      $organizationCompCB configure -state normal
      $organizationStringE configure -state normal
      $address1LF configure -state normal
      $address1CompCB configure -state normal
      $address1StringE configure -state normal
      $address2LF configure -state normal
      $address2CompCB configure -state normal
      $address2StringE configure -state normal
      $cityLF configure -state normal
      $cityCompCB configure -state normal
      $cityStringE configure -state normal
      $stateLF configure -state normal
      $stateCompCB configure -state normal
      $stateCB configure -state normal
      $zipLF configure -state normal
      $zipCompCB configure -state normal
      $zipStringE configure -state normal
      $phoneLF configure -state normal
      $phoneCompCB configure -state normal
      $phoneStringE configure -state normal
      $faxLF configure -state normal
      $faxCompCB configure -state normal
      $faxStringE configure -state normal
      $searchButtons configure -state normal
      $rightTF configure -state normal
      $clauseList configure -state normal
      $clauseButtons configure -state normal
    }
    #***************************************
    # close method -- disconnect from the database and disable all of the GUI 
    # elements
    #***************************************
    method close {} {
      set _Connection {}
      catch {$_searchStatement drop}
      set _searchStatement {}
      $clauseList delete [$clauseList items]
      $leftTF configure -state disabled
      $agregatorLF configure -state disabled
      $andRB configure -state disabled
      $orRB  configure -state disabled
      $firstnameLF configure -state disabled
      $firstnameCompCB configure -state disabled
      $firstnameStringE configure -state disabled
      $lastnameLF configure -state disabled
      $lastnameCompCB configure -state disabled
      $lastnameStringE configure -state disabled
      $organizationLF configure -state disabled
      $organizationLF configure -state disabled
      $organizationCompCB configure -state disabled
      $organizationStringE configure -state disabled
      $address1LF configure -state disabled
      $address1CompCB configure -state disabled
      $address1StringE configure -state disabled
      $address2LF configure -state disabled
      $address2CompCB configure -state disabled
      $address2StringE configure -state disabled
      $cityLF configure -state disabled
      $cityCompCB configure -state disabled
      $cityStringE configure -state disabled
      $stateLF configure -state disabled
      $stateCompCB configure -state disabled
      $stateCB configure -state disabled
      $zipLF configure -state disabled
      $zipCompCB configure -state disabled
      $zipStringE configure -state disabled
      $phoneLF configure -state disabled
      $phoneCompCB configure -state disabled
      $phoneStringE configure -state disabled
      $faxLF configure -state disabled
      $faxCompCB configure -state disabled
      $faxStringE configure -state disabled
      $searchButtons configure -state disabled
      $rightTF configure -state disabled
      $clauseList configure -state disabled
      $clauseButtons configure -state disabled
    }
    #***************************************
    # destructor -- make sure the connection is closed.
    #***************************************
    destructor {
      catch {$self close}
    }
    #***************************************
    # __CheckField method -- check to see if the selected field has a non 
    # null compare string and if so, return the clause
    #***************************************
    method _CheckField {name comp field} {
      set val "[$field cget -text]"
      if {[string length "$val"] == 0} {
	return {}
      } else {
        $field configure -text {}
	switch [$comp cget -text] {
	  contains {
	    return [list "$name LIKE ?" "%${val}%"]
	  }
	  {does not contain} {
	    return [list "$name NOT LIKE ?" "%${val}%"]
	  }
	  is {
	    return [list "$name IS ?" "$val"]
	  }
	  {is not} {
	    return [list "$name IS NOT ?" "$val"]
	  }
	}
      }
    }
    #***************************************
    # _AddClause method -- check all fields and add clauses.
    #***************************************
    method _AddClause {} {
      foreach n {firstname lastname organization address1 address2 city state zip telephone 
		 fax} \
	      c [list $firstnameCompCB $lastnameCompCB $organizationCompCB \
		      $address1CompCB $address2CompCB $cityCompCB $stateCompCB \
		      $zipCompCB $phoneCompCB $faxCompCB] \
	      f [list $firstnameStringE $lastnameStringE $organizationStringE \
		      $address1StringE $address2StringE $cityStringE $stateCB \
		      $zipStringE $phoneStringE $faxStringE] {
	if {[llength [$clauseList items]] > 0} {
          set newclause "$_Agregator "
	} else {
	  set newclause {}
	}
	set cL [$self _CheckField $n $c $f]
	if {[llength $cL] < 2} {continue}
	append newclause "[lindex $cL 0]"
	$clauseList insert end #auto \
			-text [regsub {\?} "$newclause" "[lindex $cL 1]"] \
			-data [list $newclause "[lindex $cL 1]"]
      }
    }
    #***************************************
    # dosearch method -- perform a search.
    #***************************************
    method dosearch {} {
      # If the search buttons are disabled, abort the search -- there
      # is no database connection
      set state [$searchButtons cget -state]
      if {![string equal "$state" normal]} {return}
      # Add any clauses
      $self _AddClause
      # Get clause list
      set clauses [$clauseList items]
      # Start to build SQL string
      set sql "select * from addresses"
      set arglist {}
      set argdefs {}
      # Loop over clauses, adding clauses, arglist elements, and argdef elements
      if {[llength $clauses] > 0} {
	append sql " where"
	foreach cl $clauses {
	  set cL [$clauseList itemcget $cl -data]
	  append sql " [lindex $cL 0]"
	  lappend arglist [lindex $cL 1]
	  lappend argdefs CHAR
	}
      }
      # Add order clause
      append sql " order by lastname, firstname"
#      puts stderr "*** $self dosearch: sql = $sql"
      # Drop an old statement
      catch {$_searchStatement drop}
      # Create statement object
      set _searchStatement [$_Connection statement \
				[DatabaseWidgets::statementID create] "$sql" $argdefs]
      # Call start hook, if any
      if {[string length "$options(-searchstarthook)"] > 0} {
	set result [uplevel #0 $options(-searchstarthook) $options(-callbackargs)]
#	puts stderr "*** $self dosearch: (-searchstarthook): result = $result"
	if {!$result} {return}
      }
      # Start search
      $_searchStatement execute $arglist
      # Loop over rows
      set rowcount 0
      while {[set row [$_searchStatement fetch]] != {}} {
        # Call row function
	uplevel #0 $options(-rowfunction) [list $row] $options(-callbackargs)
#	puts stderr "*** $self dosearch: (-rowfunction): row = $row"
	incr rowcount
      }
#      puts stderr "*** $self dosearch: rowcount = $rowcount"
      # Call result hook
      if {[string length "$options(-searchresulthook)"] > 0} {
	uplevel #0 $options(-searchresulthook) $rowcount $options(-callbackargs)
      }
    }
    #***************************************
    # _DeleteClause method -- delete selected clause
    #***************************************
    method _DeleteClause {} {
      # Get selected clauses
      set selection [$clauseList selection get]
      # If none selected, return
      if {[llength $selection] == 0} {return}
      # Delete selected clause
      set selection [lindex $selection 0]
      $clauseList delete $selection
      # Check first clause and remove agregator if any
      set first [lindex [$clauseList items] 0]
      set toptext "[$clauseList itemcget $first -text]"
      set topCl    [$clauseList itemcget $first -data]
      if {[regexp {^(AND|OR) } "$toptext"] > 0} {
	regsub {^(AND|OR) } "$toptext" {} toptext
	set clause [lindex $topCl 0]
	regsub {^(AND|OR) } "$clause" {} clause
	set topCl [list $clause [lindex $topCl 1]]
	$clauseList itemconfigure $first -text "$toptext" -data $topCl
      }
    }
    #***************************************
    # _NewSearch method -- delete all clauses.
    #***************************************
    method _NewSearch {} {
      $clauseList delete [$clauseList items]
    }
  }
  #***************************************
  # Macro to create a base Search frame in a dialog
  #***************************************
  snit::macro DatabaseWidgets::SearchDialogConstructor {helptopic btext body} {
    # Common options, both delegated to the hull (a Dialog)
    delegate option -title to hull;#	Title
    delegate option -parent to hull;#	Parent
    component searchFR;#		Search frame
    # Stock constructor body: Build the standard dialog with a search frame.
    # Other GUI elements are added below the search frame.
    set cbody {
	installhull using Dialog::create -modal local \
			-bitmap question -default 0 -cancel 1 -side bottom \
			-parent .
        $self configurelist $args
        $hull add -name search -text %%BTEXT%% -command [mymethod _Search]
	$hull add -name cancel -text Cancel -command [mymethod _Cancel]
	$hull add -name help -text Help \
			     -command "BWHelp::HelpTopic %%HELPTOPIC%%"
	set frame [$hull getframe]
	install searchFR using DatabaseWidgets::SearchFrame \
				$frame.searchFR \
				-rowfunction [mymethod _RowFunction]\
				-searchresulthook [mymethod _ResultHook]\
				-searchstarthook [mymethod _StartHook]\
				-includesearch no
	pack $searchFR -expand yes -fill both
    }
    # Substitute in custom values:
    regsub -all {%%HELPTOPIC%%} "$cbody" "$helptopic" cbody
    regsub -all {%%BTEXT%%} "$cbody" "$btext" cbody
    # Append the rest of the body
    append cbody "$body"
    # Build constructor
    constructor {args} "$cbody"
    # Delegate the open and close methods to the search frame
    delegate method open to searchFR
    delegate method close to searchFR
    # Search button binding
    method _Search {} {
      $searchFR dosearch
      $hull withdraw
      return [$hull enddialog search]
    }
    # Cancel button binding
    method _Cancel {} {
      $hull withdraw
      return [$hull enddialog cancel]
    }
    # Draw method
    method draw {args} {
      $self configurelist $args
      wm transient [winfo toplevel $win] [$hull cget -parent]
      $self _DrawHook;# Hook function for drawing the dialog
      return [$hull draw]
    }
  }
  #***************************************
  # Export selected rows to a CSV (Comma Separated Value) file
  #***************************************
  snit::widgetadaptor ExportCSVDialog {
    # CSV File types
    typevariable CSVFileTypes {
      {{CSV Files} {.csv} TEXT}
      {{All Files} *      TEXT}
    }
    #*************************************
    # Quote CSV value field
    #*************************************
    typemethod QuoteCSV {string} {
      if {[regexp {[,"]} "$string"] > 0} {
	regsub -all {["]} "$string" {\\&} string
	return "\"$string\""
      } else {
	return "$string"
      }
    }
    # Additional components:
    component fileLF;#		File name label frame
    component   fileE;#		File name entry
    component   fileB;#		File browse button
    # Instance variable:
    variable _FP {};#		Open file
    # Search frame dialog constructor:
    DatabaseWidgets::SearchDialogConstructor ExportCSVDialog Export {
      # Output file:
      install fileLF using LabelFrame::create [$hull getframe].fileLF \
				-text "Export File:"\
				-width $::DatabaseWidgets::LabelWidth
      pack $fileLF -fill x
      set fileLFfr [$fileLF getframe]
      install fileE using Entry::create $fileLFfr.fileE
      pack $fileE -expand yes -fill x -side left
      install fileB using Button::create $fileLFfr.fileB \
			-text Browse\
			-command [mymethod _Browse]
      pack $fileB -side right
    }
    #***************************************
    # Browse for an output (CSV) file
    #***************************************
    method _Browse {} {
      set existingFile "[$fileE cget -text]"
      set initdir "[file dirname $existingFile]"
      set newfile [tk_getSaveFile -title "File to export to" -parent $win \
			-initialfile "$existingFile" -initialdir "$initdir" \
			-defaultextension {.csv} -filetypes $CSVFileTypes]
      if {[string length "$newfile"] > 0} {
	$fileE configure -text "$newfile"
      }
    }
    #***************************************
    # Unused Draw Hook
    #***************************************
    method _DrawHook {} {}
    #***************************************
    # Start hook: Open output file
    #***************************************
    method _StartHook {} {
      set filename "[$fileE cget -text]"
      if {[catch {open "$filename" w} _FP]} {
	tk_messageBox -type ok -icon error -message "Could not open $filename: $_FP"
	set _FP {}
        return false
      } else {
        return true
      }
    }
    #*************************************
    # Export one CSV row
    #*************************************
    method _RowFunction {row} {
      set needcomma no
      foreach ele $row {
	if {$needcomma} {puts -nonewline $_FP {,}}
	puts -nonewline $_FP [$type QuoteCSV "$ele"]
	set needcomma yes
      }
      puts $_FP {}
    }
    #***************************************
    # Result Hook: close output file
    #***************************************
    method _ResultHook {count} {
      close $_FP
      set _FP {}
      tk_messageBox -type ok -icon info -message "$count rows exported"
    }
  }
  #***************************************
  # Export to LaTeX Envelope file
  #***************************************
  snit::widgetadaptor ExportEnvDialog {
    # LaTeX file types
    typevariable TeXFileTypes {
      {{LaTeX Files} {.tex .ltx} TEXT}
      {{All Files} *      TEXT}
    }
    #*************************************
    # Quote text string
    #*************************************
    typemethod QuoteTeX {string} {
      regsub -all {[_&\$]} "$string" {\\&} string
      return "$string"
    }
    # Additional components:
    component fileLF;#		File name label frame
    component   fileE;#		File name entry
    component   fileB;#		File browse button
    # Instance variables
    variable _FP {};#		Open file
    # Search frame dialog constructor:
    DatabaseWidgets::SearchDialogConstructor ExportEnvDialog Export {
      # Output file:
      install fileLF using LabelFrame::create [$hull getframe].fileLF \
				-text "Export File:"\
				-width $::DatabaseWidgets::LabelWidth
      pack $fileLF -fill x
      set fileLFfr [$fileLF getframe]
      install fileE using Entry::create $fileLFfr.fileE
      pack $fileE -expand yes -fill x -side left
      install fileB using Button::create $fileLFfr.fileB \
			-text Browse\
			-command [mymethod _Browse]
      pack $fileB -side right
    }
    #***************************************
    # Browse for an output (LaTeX) file
    #***************************************
    method _Browse {} {
      set existingFile "[$fileE cget -text]"
      set initdir "[file dirname $existingFile]"
      set newfile [tk_getSaveFile -title "File to export to" -parent $win \
			-initialfile "$existingFile" -initialdir "$initdir" \
			-defaultextension {.tex} -filetypes $TeXFileTypes]
      if {[string length "$newfile"] > 0} {
	$fileE configure -text "$newfile"
      }
    }
    #***************************************
    # Unused Draw Hook
    #***************************************
    method _DrawHook {} {}
    #*************************************
    # Export envelope row
    #*************************************
    method _RowFunction {row} {
      # Get fields
      set sal "[lindex $row 1]"
      set fn  "[lindex $row 2]"
      set ln  "[lindex $row 3]"
      set suf  [string trim "[lindex $row 4]"]
      if {[string length "$sal"] > 0} {
	set name "$sal "
      } else {
	set name {}
      }
      append name "$fn $ln"
      if {[string length "$suf"] > 0} {append name ", $suf"}
      set name [$type QuoteTeX "$name"]
      set org  [$type QuoteTeX [lindex $row 5]]
      set addr1 [$type QuoteTeX [lindex $row 6]]
      set addr2 [$type QuoteTeX [lindex $row 7]]
      set csz [$type QuoteTeX "[lindex $row 8], [lindex $row 9] [lindex $row 10]"]
      set zip [$type QuoteTeX [lindex $row 10]]

      # Output envelope environment
      puts $_FP {\begin{envelope}}
      puts $_FP "$name \\\\"
      if {[string length "$org"] > 0} {
	puts $_FP "$org \\\\"
      }
      puts $_FP "$addr1 \\\\"
      if {[string length "$addr2"] > 0} {
	puts $_FP "$addr2 \\\\"
      }
      puts $_FP "$csz"
      puts $_FP "\\ZipBar{$zip}"
      puts $_FP {\end{envelope}}
    }
    #***************************************
    # Start hook: Open output file and write preamble
    #***************************************
    method _StartHook {} {
      set filename "[$fileE cget -text]"
      if {[catch {open "$filename" w} _FP]} {
	tk_messageBox -type ok -icon error -message "Could not open $filename: $_FP"
	set _FP {}
        return false
      } else {
	# Output LaTeX preamble
	puts $_FP {\documentclass[12pt]{article}
\usepackage{env}
\nofiles
\begin{document}

\sf
}
	return true
      }
    }
    #***************************************
    # Result Hook: write trailer and close output file
    #***************************************
    method _ResultHook {count} {
      puts $_FP {\end{document}}
      close $_FP
      set _FP {}
      tk_messageBox -type ok -icon info -message "$count rows exported"
    }
  }
  #***************************************
  # Export to LaTeX Letter environments (form letter)
  #***************************************
  snit::widgetadaptor ExportLetterDialog {
    # LaTeX file types
    typevariable TeXFileTypes {
      {{LaTeX Files} {.tex .ltx} TEXT}
      {{All Files} *      TEXT}
    }
    #*************************************
    # Quote text string
    #*************************************
    typemethod QuoteTeX {string} {
      regsub -all {[_&\$]} "$string" {\\&} string
      return "$string"
    }
    # Additional components:
    component fileLF;#		File name label frame
    component   fileE;#		File name entry
    component   fileB;#		File browse button
    component templatefileLF;#	Template file name label frame
    component   templatefileE;#	Template file name entry
    component   templatefileB;#	Template file browse button
    # Instance variables
    variable _FP {};#		Open file
    variable _Template;#	Template letter
    # Search frame dialog constructor:
    DatabaseWidgets::SearchDialogConstructor ExportLetterDialog Export {
      # Output file:
      install fileLF using LabelFrame::create [$hull getframe].fileLF \
				-text "Export File:"\
				-width $::DatabaseWidgets::LabelWidth
      pack $fileLF -fill x
      set fileLFfr [$fileLF getframe]
      install fileE using Entry::create $fileLFfr.fileE
      pack $fileE -expand yes -fill x -side left
      install fileB using Button::create $fileLFfr.fileB \
			-text Browse\
			-command [mymethod _Browse]
      pack $fileB -side right
      # Template (form) letter file
      install templatefileLF using LabelFrame::create [$hull getframe].templatefileLF \
				-text "Template File:"\
				-width $::DatabaseWidgets::LabelWidth
      pack $templatefileLF -fill x
      set templatefileLFfr [$templatefileLF getframe]
      install templatefileE using Entry::create $templatefileLFfr.templatefileE
      pack $templatefileE -expand yes -fill x -side left
      install templatefileB using Button::create $templatefileLFfr.templatefileB \
			-text Browse\
			-command [mymethod _TemplateBrowse]
      pack $templatefileB -side right
    }
    #***************************************
    # Browse for an output (LaTeX) file
    #***************************************
    method _Browse {} {
      set existingFile "[$fileE cget -text]"
      set initdir "[file dirname $existingFile]"
      set newfile [tk_getSaveFile -title "File to export to" -parent $win \
			-initialfile "$existingFile" -initialdir "$initdir" \
			-defaultextension {.tex} -filetypes $TeXFileTypes]
      if {[string length "$newfile"] > 0} {
	$fileE configure -text "$newfile"
      }
    }
    #***************************************
    # Browse for an input (LaTeX) file -- form letter
    #***************************************
    method _TemplateBrowse {} {
      set existingFile "[$templatefileE cget -text]"
      set initdir "[file dirname $existingFile]"
      set newfile [tk_getOpenFile -title "Template File to use" -parent $win \
			-initialfile "$existingFile" -initialdir "$initdir" \
			-defaultextension {.tex} -filetypes $TeXFileTypes]
      if {[string length "$newfile"] > 0} {
	$templatefileE configure -text "$newfile"
      }
    }
    #***************************************
    # Unused Draw Hook
    #***************************************
    method _DrawHook {} {}
    #*************************************
    # Output form letter 
    #*************************************
    method _RowFunction {row} {
      # Get fields
      set sal [$type QuoteTeX "[lindex $row 1]"]
      set fn  [$type QuoteTeX "[lindex $row 2]"]
      set ln  [$type QuoteTeX "[lindex $row 3]"]
      set suf [$type QuoteTeX [string trim "[lindex $row 4]"]]
      if {[string length "$sal"] > 0} {
	set name "$sal "
      } else {
	set name {}
      }
      append name "$fn $ln"
      if {[string length "$suf"] > 0} {append name ", $suf"}
      set org   [$type QuoteTeX "[lindex $row 5]"]
      set addr1 [$type QuoteTeX "[lindex $row 6]"]
      set addr2 [$type QuoteTeX "[lindex $row 7]"]
      set city  [$type QuoteTeX "[lindex $row 8]"]
      set state [$type QuoteTeX "[lindex $row 9]"]
      set zip   [$type QuoteTeX "[lindex $row 10]"]
      set csz   "$city, $state $zip"
      set phone [$type QuoteTeX "[lindex $row 11]"]
      set fax   [$type QuoteTeX "[lindex $row 12]"]

      # Output letter environment
      puts -nonewline $_FP {\begin{letter}}
      puts $_FP " \{"
      puts $_FP "$name \\\\"
      if {[string length "$org"] > 0} {
	puts $_FP "$org \\\\"
      }
      puts $_FP "$addr1 \\\\"
      if {[string length "$addr2"] > 0} {
	puts $_FP "$addr2 \\\\"
      }
      puts $_FP "$csz"
      puts $_FP "\}"
      regsub -all {%salutation}   "$_Template" "$sal" letter
      regsub -all {%firstname}    "$letter" "$fn" letter
      regsub -all {%lastname}     "$letter" "$ln" letter
      regsub -all {%suffix}       "$letter" "$suf" letter
      regsub -all {%organization} "$letter" "$org" letter
      regsub -all {%address1}     "$letter" "$addr1" letter
      regsub -all {%address2}     "$letter" "$addr2" letter
      regsub -all {%city}         "$letter" "$city" letter
      regsub -all {%state}        "$letter" "$state" letter
      regsub -all {%zip}          "$letter" "$zip" letter
      regsub -all {%telephone}    "$letter" "$phone" letter
      regsub -all {%fax}          "$letter" "$fax" letter
      puts $_FP "$letter"
      puts $_FP {\end{letter}}
    }
    #***************************************
    # Start hook: Open output file and write preamble
    # Also load in template (form) letter
    #***************************************
    method _StartHook {} {
      set templatefilename "[$templatefileE cget -text]"
      if {[string equal "$templatefilename" {}]} {
        set templatepreamble {\usepackage{times}
\name{}
\address{}
}
	set _Template {
\\opening{Dear %salutation %lastname:}

\\closing{Sincerely}
}
      } else {
	if {[catch {open $templatefilename r} tempFP]} {
	  tk_messageBox -type ok -icon error -message "Could not open $templatefilename: $tempFP"
	  return false
	}
	set templatepreamble {}
	set _Template {}
	while {[gets $tempFP line] >= 0} {
	  if {[string equal -nocase "$line" {%%endofpreamble%%}]} {break}
	  append templatepreamble "$line\n"
	}
	set _Template "[read $tempFP]"
	close $tempFP
      }
      set filename "[$fileE cget -text]"
      if {[catch {open "$filename" w} _FP]} {
	tk_messageBox -type ok -icon error -message "Could not open $filename: $_FP"
	set _FP {}
        return false
      } else {
	# Output LaTeX preamble
	puts $_FP {\documentclass[12pt]{letter}}
	puts $_FP "$templatepreamble"
        puts $_FP {
\begin{document}
}
	return true
      }
    }
    #***************************************
    # Result Hook: write trailer and close output file
    #***************************************
    method _ResultHook {count} {
      puts $_FP {\end{document}}
      close $_FP
      set _FP {}
      tk_messageBox -type ok -icon info -message "$count rows exported"
    }
  }
}

#***************************************
# (Re-)Initialize datebase: drop and rebuild Address Table
#***************************************
proc DatabaseWidgets::CreateAddressTable {Connection} {
  variable TableName
  variable TableSpecification

  # Check for existing table
  if {[catch {$Connection tables $TableName} tableinfo]} {
    tk_messageBox -type ok -icon error -message "Open a connection to a database first!"
    return
  }
  # Table already exists?  Verify clobbering it...
  if {[llength $tableinfo] > 0} {
    if {[string equal [tk_messageBox -type yesno -default no -icon question \
	-message "Are you sure you want to re-initialize your database?"] {no}]} {
      return
    }
    $Connection {drop table $TableName}
  }
  # Create address table.
  set result [catch {$Connection "create TABLE $TableName $TableSpecification"} error]
  if {$result} {tk_messageBox -type ok -icon error -message "$error"}
}

#***************************************
# Delete a row from the address table
#***************************************
proc DatabaseWidgets::DeleteAddress {Connection id} {
  variable TableName

  $Connection "delete from $TableName where id = ?" [list $id]
}

package provide DatabaseWidgets 1.0

