#* 
#* ------------------------------------------------------------------
#* BWStdMenuBar.tcl - BWidget version of StdMenuBar
#* Created by Robert Heller on Sun Feb  5 14:36:04 2006
#* ------------------------------------------------------------------
#* Modification History: $Log$
#* Modification History: Revision 1.2  2007/04/19 17:23:23  heller
#* Modification History: April 19 Lock Down
#* Modification History:
#* Modification History: Revision 1.1  2006/02/06 00:20:44  heller
#* Modification History: Start converting to BWidget from Tix
#* Modification History:
#* Modification History: Revision 1.1  2002/07/28 14:03:50  heller
#* Modification History: Add it copyright notice headers
#* Modification History:
#* ------------------------------------------------------------------
#* Contents:
#* ------------------------------------------------------------------
#*  
#*     Model RR System, Version 2
#*     Copyright (C) 1994,1995,2002-2005  Robert Heller D/B/A Deepwoods Software
#* 			51 Locke Hill Road
#* 			Wendell, MA 01379-9728
#* 
#*     This program is free software; you can redistribute it and/or modify
#*     it under the terms of the GNU General Public License as published by
#*     the Free Software Foundation; either version 2 of the License, or
#*     (at your option) any later version.
#* 
#*     This program is distributed in the hope that it will be useful,
#*     but WITHOUT ANY WARRANTY; without even the implied warranty of
#*     MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
#*     GNU General Public License for more details.
#* 
#*     You should have received a copy of the GNU General Public License
#*     along with this program; if not, write to the Free Software
#*     Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
#* 
#*  
#* 

#@Chapter:BWStdMenuBar.tcl -- Create standard menubars
#@Label:BWStdMenuBar.tcl
#$Id: snitStdMenuBar.tcl 2558 2017-03-12 20:47:36Z heller $
# This file contains code to create a standard Motif style menubar.
# A standard menubar contains ``File'', ``Edit'', ``View'', ``Options'',
# and ``Help'' pulldown menus.  The ``File'', ``Edit'', and ``Help'' menus
# have standard menu items.
#
# The menubars and menus generated by the procedures in this file fill the
# standards set forth in the Motif Style Guide.

package require snit
package require gettext

snit::type StdMenuBar {
# StdMenuBar snit type
# [index] StdMenuBar!type

  pragma -hastypeinfo    no
  pragma -hastypedestroy no
  pragma -hasinstances   no

  typevariable _menu .menu
  # The name of the menu widget (not actually used).

  typevariable _std_file_menu 
  # The standard basic File menu.

  typevariable _std_edit_menu
  # The standard basic Edit menu.

  typevariable _std_view_menu
  # The standard basic View menu.

  typevariable _std_options_menu 
  # The standard basic Options menu.

  typevariable _std_help_menu 
  # The standard basic Help menu.

  typeconstructor {
      set _std_file_menu [list [_m "Menu|&File"] {file:menu} {file} 0]
      lappend _std_file_menu_cs \
            [list command [_m "Menu|File|&New"]     {file:new} ""     {Ctrl n}]
      lappend _std_file_menu_cs \
            [list command [_m "Menu|File|&Open..."] {file:open} "" {Ctrl o}]
      lappend _std_file_menu_cs \
            [list command [_m "Menu|File|&Save"]    {file:save} "" {Ctrl s}]
      lappend _std_file_menu_cs \
            [list command [_m "Menu|File|Save &As..."] {file:save} "" {Ctrl a}]
      lappend _std_file_menu_cs \
            [list command [_m "Menu|File|&Close"] {file:close} [_ "Close the application"] {}]
      lappend _std_file_menu_cs \
            [list command [_m "Menu|File|E&xit"] {file:exit} [_ "Exit the application"] {}]
      lappend _std_file_menu_cs $_std_file_menu_cs
      set _std_edit_menu [list [_m "Menu|&Edit"] {edit} {edit} 0]
      lappend _std_edit_menu_cs \
            [list command [_m "Menu|Edit|&Undo"] {edit:undo} [_ "Undo last change"] {Ctrl z}]
      lappend _std_edit_menu_cs \
            [list command [_m "Menu|Edit|Cu&t"] {edit:cut edit:havesel} [_ "Cut selection to the paste buffer"] {Ctrl x} -command [mytypemethod EditCut]]
      lappend _std_edit_menu_cs \
            [list command [_m "Menu|Edit|&Copy"] {edit:copy edit:havesel} [_ "Copy selection to the paste buffer"] {Ctrl c} -command [mytypemethod EditCopy]]
      lappend _std_edit_menu_cs \
            [list command [_m "Menu|Edit|C&lear"] {edit:clear edit:havesel} [_ "Clear selection"] {} -command [mytypemethod EditClear]]
      lappend _std_edit_menu_cs \
            [list command [_m "Menu|Edit|&Delete"] {edit:delete edit:havesel} [_ "Delete selection"] {Ctrl d} -command [mytypemethod EditClear]]
      lappend _std_edit_menu_cs \
            [list separator]
      lappend _std_edit_menu_cs \
            [list command [_m "Menu|Edit|Select All"] {edit:selectall} [_ "Select everything"] {} -command [mytypemethod EditSelectAll]]
      lappend _std_edit_menu_cs \
            [list command [_m "Menu|Edit|De-select All"] {edit:deselectall edit:havesel} [_ "Select nothing"] {} -command [mytypemethod EditSelectNone]]
      lappend _std_edit_menu $_std_edit_menu_cs
      set _std_view_menu [list [_m "Menu|&View"] {view} {view} 0 {}]
      set _std_options_menu [list [_m "Menu|&Options"] {options} {options} 0 {}]
      set _std_help_menu [list [_m "Menu|&Help"] {help} {help} 0]
      lappend _std_help_menu_cs \
            [list command [_m "Menu|Help|On &Help..."] {help:help} [_ "Help on help"] {}]
      lappend _std_help_menu_cs \
            [list command [_m "Menu|Help|On &Keys..."] {help:keys} [_ "Help on keyboard accelerators"] {}]
      lappend _std_help_menu_cs \
            [list command [_m "Menu|Help|&Index..."] {help:index} [_ "Help index"] {}]
      lappend _std_help_menu_cs \
            [list command [_m "Menu|Help|&Tutorial..."] {help:tutorial} [_ "Tutorial"] {}]
      lappend _std_help_menu_cs \
            [list command [_m "Menu|Help|On &Version"] {help:version} [_ "Version"] {}]
      lappend _std_help_menu_cs \
            [list command [_m "Menu|Help|Warranty"] {help:warranty} [_ "Warranty"] {}]
      lappend _std_help_menu_cs \
            [list command [_m "Menu|Help|Copying"] {help:copying} [_ "Copying"] {}]
      lappend _std_help_menu $_std_help_menu_cs
  }
  
  typemethod MakeMenu {args} {
# Make a complete Standard Menu for the BW MainWindow widget.
# <in> args -- Menu overwrite options.  This is an alternating list of options
#	       and values. Each option is either one of the standard menu
#	       items (-file, -edit, -view, -options, or -help), in which
#	       case the value replaces the standard menu, or it is another 
#	       option, which introduces a new menu item to be added after the
#	       options menu item, after any other new menu item.  The value
#	       for these options is for a single menu item, as described in
#	       the documentation for the -menu option for the BWidget
#	       MainFrame: {menuname tags menuId tearoff menuentries...}.
# [index] StdMenuBar MakeMenu!procedure

  set menu [list -file $_std_file_menu \
  		 -edit $_std_edit_menu \
		 -view $_std_view_menu \
		 -options $_std_options_menu \
		 -help $_std_help_menu \
		]
  #puts stderr "*** StdMenuBar::MakeMenu: menu = $menu"
  foreach {option value} $args {
    set index [lsearch -exact $menu $option]
    #puts stderr "*** StdMenuBar::MakeMenu: index = $index"
    #puts stderr "*** StdMenuBar::MakeMenu: option = $option"
    #puts stderr "*** StdMenuBar::MakeMenu: value = $value"
    if {$index < 0} {
      set hindex [lsearch -exact $menu -help]
      if {$hindex < 0} {
	lappend menu $option $value
      } else {
	set menu [lreplace $menu $hindex $hindex $option $value -help]
      }
    } else {
      set menu [lreplace $menu [expr $index + 1] [expr $index + 1] $value]
    }
  }
  #puts stderr "*** StdMenuBar::MakeMenu: menu = $menu"
  set result {}
  foreach {option value} $menu {
    eval [concat lappend result $value]
  }
  #puts stderr "*** StdMenuBar::MakeMenu: result = $result"
  return $result
}

typemethod EditCut {} {
# Handle the Cut item on the Edit menu.
# [index] StdMenuBar::EditCut!procedure

  set f "[::focus]"
  if {[string equal "$f" {}]} {return}
  catch "event generate $f <<Cut>>"
}
    
typemethod EditCopy {} {
# Handle the Copy item on the Edit menu.
# [index] StdMenuBar::EditCopy!procedure

  set f "[::focus]"
  if {[string equal "$f" {}]} {return}
  catch "event generate $f <<Copy>>"
}

typemethod EditPaste {} {
# Handle the Paste item on the Edit menu.
# [index] StdMenuBar::EditPaste!procedure

  set f "[::focus]"
  if {[string equal "$f" {}]} {return}
  catch "event generate $f <<Paste>>"
}

typemethod EditClear {} {
# Handle the Clear item on the Edit menu.
# [index] StdMenuBar::EditClear!procedure

  set f "[::focus]"
  if {[string equal "$f" {}]} {return}
  catch "event generate $f <<Clear>>"
}
typemethod EditSelectAll {} {
    # Handle the Select All item on the Edit menu.
    # [index] StdMenuBar::EditSelectAll!procedure
    
    set f "[::focus]"
    #puts stderr "*** $type EditSelectAll: f is $f"
    if {[string equal "$f" {}]} {return}
    catch {
        switch [winfo class $f] {
            Spinbox -
            Entry -
            TEntry {
                $f selection range 0 end
            }
            Text {
                $f tag add sel 1.0 end
            }
        }
    }
}
typemethod EditSelectNone {} {
    # Handle the Select None item on the Edit menu.
    # [index] StdMenuBar::EditSelectNone!procedure
    
    set f "[::focus]"
    #puts stderr "*** $type EditSelectNone: f is $f"
    if {[string equal "$f" {}]} {return}
    switch [winfo class $f] {
        Spinbox -
        Entry -
        TEntry {
            $f selection clear
        }
        Text {
            $f tag remove sel 1.0 end
        }
    }
}

}


snit::widgetadaptor StdEditContextMenu {
    method bind {tag} {
        bind $tag <3> [mymethod _postEditMenu %W %X %Y]
    }
    variable top {}
    variable tags -array {}
    variable tagstate -array {}
    variable menutags -array {}
    delegate method * to hull except {unpost add entrycget entryconfigure}
    typevariable _editmenu {
        {command "[_m {Menu|Edit|Cu&t}]" {edit:cut edit:havesel} "[_ {Cut selection to the paste buffer}]" {Ctrl x} -command {StdMenuBar EditCut} -state disabled}
        {command "[_m {Menu|Edit|&Copy}]" {edit:copy edit:havesel} "[_ {Copy selection to the paste buffer}]" {Ctrl c} -command {StdMenuBar EditCopy} -state disabled}
        {command "[_m {Menu|Edit|&Paste}]" {edit:paste} "[_ {Paste selection from the paste buffer}]" {Ctrl c} -command {StdMenuBar EditPaste} -state disabled}
        {command "[_m {Menu|Edit|C&lear}]" {edit:clear edit:havesel} "[_ {Clear selection}]" {} -command {StdMenuBar EditClear} -state disabled}
        {command "[_m {Menu|Edit|&Delete}]" {edit:delete edit:havesel} "[_ {Delete selection}]" {Ctrl d}  -command {StdMenuBar EditClear} -state disabled}
        {separator}
        {command "[_m {Menu|Edit|Select All}]" {edit:selectall} "[_ {Select everything}]" {} -command {StdMenuBar EditSelectAll}}
        {command "[_m {Menu|Edit|De-select All}]" {edit:deselectall edit:havesel} "[_ {Select nothing}]" {} -command {StdMenuBar EditSelectNone} -state disabled}
    }
    constructor {args} {
        set menuitems [subst $_editmenu]
        installhull using menu -tearoff no \
              -postcommand [mymethod _postcommand]              
        set top [winfo parent $win]
        $self _create_entries $hull $menuitems
        #$self configurelist $args
    }
    method setmenustate { tag state } {
        # We need a more sophisticated state system.
        # The original model was this:  each menu item has a list of tags;
        # whenever any one of those tags changed state, the menu item did too.
        # This makes it hard to have items that are enabled only when both tagA and
        # tagB are.  The new model therefore only sets the menustate to enabled
        # when ALL of its tags are enabled.
        
        # First see if this is a real tag
        if { [info exists tagstate($tag)] } {
            if { ![string equal $state "disabled"] } {
                set tagstate($tag) 1
            } else {
                set tagstate($tag) 0
            }
            foreach {menu entry} $tags($tag) {
                set expression "1"
                foreach menutag $menutags([list $menu $entry]) {
                    append expression " && $tagstate($menutag)"
                }
                if { [expr $expression] } {
                    set state normal
                } else {
                    set state disabled
                }
                $menu entryconfigure $entry -state $state
            }
        }
        return
    }
    method _create_entries {menu entries } {
        #puts stderr "*** _create_entries $menu $entries"
        set count      [$menu cget -tearoff]
        set registered 0
        foreach entry $entries {
            #puts stderr "*** _create_entries: entry = {$entry}"
            set len  [llength $entry]
            #puts stderr "*** _create_entries: len = $len"
            set _type [lindex $entry 0]
            #puts stderr "*** _create_entries: _type = $_type"
            if { [string equal $_type "separator"] } {
                $menu add separator
                incr count
                continue
            }
            # entry name and tags
            set opt  [_parse_name [lindex $entry 1]]
            #puts stderr "*** _create_entries: opt = $opt"
            set _tags [lindex $entry 2]
            #puts stderr "*** _create_entries: _tags = $_tags"
            foreach tag $_tags {
                lappend tags($tag) $menu $count
                # ericm@scriptics:  Add a tagstate tracker
                if { ![info exists tagstate($tag)] } {
                    set tagstate($tag) 1
                }
            }
            # ericm@scriptics:  Add mapping from menu items to tags
            set menutags([list $menu $count]) $_tags
            if {[string equal $_type "cascade"] || [string equal $_type "cascad"]} {
                set _menuid  [lindex $entry 3]
                set tearoff [lindex $entry 4]
                set submenu $menu.menu$count
                if {$len > 6} {
                    set cascadeopts [lrange $entry 5 end-1]
                } else {
                    set cascadeopts {}
                }
                eval [list $menu add cascade] $opt [list -menu $submenu] $cascadeopts
                eval [list menu $submenu -tearoff $tearoff]
                if { [string length $_menuid] } {
                    # menu has identifier
                    set menuid($_menuid) $submenu
                }
                $self _create_entries $submenu [lindex $entry end]
                incr count
                continue
            }
            
            # entry help description
            set desc [lindex $entry 3]
            #puts stderr "*** $self _create_entries: desc = $desc"
            if { [string length $desc] } {
                #if { !$registered } {
                #    DynamicHelp register $menu menu $options(-textvariable)
                #    set registered 1
                #}
                DynamicHelp register $menu menuentry $count $desc
            }

            # entry accelerator
            #puts stderr "*** $self _create_entries: \[lindex \$entry 4\] = {[lindex $entry 4]}"
            set accel [_parse_accelerator [lindex $entry 4]]
            #puts stderr "*** $self _create_entries: accel = \{$accel\}"
            if { [llength $accel] } {
                lappend opt -accelerator [lindex $accel 0]
                bind $top [lindex $accel 1] [list $menu invoke $count]
            }

            # user options
            set useropt [lrange $entry 5 end]
            if { [string equal $_type "command"] ||
                [string equal $_type "radiobutton"] ||
                [string equal $_type "checkbutton"] } {
                eval [list $menu add $_type] $opt $useropt
            } else {
                return -code error "invalid menu type \"$_type\""
            }
            incr count
        }
    }
    
    proc _parse_name {menuname} {
        set idx [string first "&" $menuname]
        if { $idx == -1 } {
            return [list -label $menuname]
        } else {
            set beg [string range $menuname 0 [expr {$idx-1}]]
            set end [string range $menuname [expr {$idx+1}] end]
            append beg $end
            return [list -label $beg -underline $idx]
        }
    }
    proc _parse_accelerator { desc } {
        if { [llength $desc] == 1 } {
            set seq None
            set key [string tolower [lindex $desc 0]]
            # If the key is an F key (ie, F1, F2, etc), it has to be capitalized
            if {[regexp {^f([1-9]|([12][0-9]|3[0-5]))$} $key]} {
                set key [string toupper $key]
            }
        } elseif { [llength $desc] == 2 } {
            set seq [lindex $desc 0]
            set key [string tolower [lindex $desc 1]]
            # If the key is an F key (ie, F1, F2, etc), it has to be capitalized
            if {[regexp {^f([1-9]|([12][0-9]|3[0-5]))$} $key]} {
                set key [string toupper $key]
            }
        } else {
            return {}
        }
        switch -- $seq {
            None {
                set accel "[string toupper $key]"
                set event "<Key-$key>"
            }
            Ctrl {
                set accel "Ctrl+[string toupper $key]"
                set event "<Control-Key-$key>"
            }
            Alt {
                set accel "Alt+[string toupper $key]"
                set event "<Alt-Key-$key>"
            }
            CtrlAlt {
                set accel "Ctrl+Alt+[string toupper $key]"
                set event "<Control-Alt-Key-$key>"
            }
            default {
                return -code error "invalid accelerator code $seq"
            }
        }
        return [list $accel $event]
    }
    method _postEditMenu {w XX YY} {
        focus $w
        #puts stderr "$self _postEditMenu: \[winfo class $w\] is [winfo class $w]"
        
        if {[catch {selection get}]} {
            set selectionavailable no
        } else {
            set selectionavailable yes
        }
        #puts stderr "$self _postEditMenu: selectionavailable is $selectionavailable"
        switch [winfo class $w] {
            Spinbox -
            Entry -
            TEntry {
                if {[$w selection present]} {
                    $self setmenustate edit:havesel normal
                } else {
                    $self setmenustate edit:havesel disabled
                }
                #puts stderr "$self _postEditMenu: \[$w cget -state\] is [$w cget -state]"
                if {[$w cget -state] eq "normal" && $selectionavailable} {
                    $self setmenustate edit:paste normal
                } else {
                    $self setmenustate edit:paste disabled
                }
            }
            Text {
                if {[$w tag ranges sel] ne {}} {
                    $self setmenustate edit:havesel normal
                } else {
                    $self setmenustate edit:havesel disabled
                }
                if {[lsearch -exact [bindtags $w] "ROText"] < 0 && $selectionavailable} {
                    $self setmenustate edit:paste normal
                } else {
                    $self setmenustate edit:paste disabled
                }
            }
        }
        $hull post $XX $YY
    }
    variable oldgrab
    variable oldgrabstate
    method _postcommand {} {
        set oldgrab [grab current]
        if {$oldgrab ne {}} {
            set oldgrabstate [grab status $oldgrab]
        }
        grab release $oldgrab
    }
    method unpost {} {
        if {$oldgrab ne {}} {
            switch $oldgrabstate {
                local {grab $oldgrab}
                global {grab -global $oldgrab}
            }
        }
        $hull unpost
    }
    
}



package provide snitStdMenuBar 1.0
  




