/* 
 * ------------------------------------------------------------------
 * Azatrax.h - Azatrax Base Class
 * Created by Robert Heller on Sun Jun 24 11:14:03 2012
 * ------------------------------------------------------------------
 * Modification History: $Log$
 * Modification History: Revision 1.1  2002/07/28 14:03:50  heller
 * Modification History: Add it copyright notice headers
 * Modification History:
 * ------------------------------------------------------------------
 * Contents:
 * ------------------------------------------------------------------
 *  
 *     Model RR System, Version 2
 *     Copyright (C) 1994,1995,2002-2012  Robert Heller D/B/A Deepwoods Software
 * 			51 Locke Hill Road
 * 			Wendell, MA 01379-9728
 * 
 *     This program is free software; you can redistribute it and/or modify
 *     it under the terms of the GNU General Public License as published by
 *     the Free Software Foundation; either version 2 of the License, or
 *     (at your option) any later version.
 * 
 *     This program is distributed in the hope that it will be useful,
 *     but WITHOUT ANY WARRANTY; without even the implied warranty of
 *     MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *     GNU General Public License for more details.
 * 
 *     You should have received a copy of the GNU General Public License
 *     along with this program; if not, write to the Free Software
 *     Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 * 
 *  
 */

#ifndef _AZATRAX_H_
#define _AZATRAX_H_

#include "config.h"

#ifdef HAVE_LIBUSB1
#include <libusb.h>
#endif

/** @defgroup Azatrax Azatrax
  * @brief Azatrax C++ LibUSB 1.0 Interface.
  *
  * This library implements a class that interfaces to Azatrax's MRD2-S,
  * MRD2-U, SL2, and SR4 modules.  Both the MRD2-S and the MRD2-U contain 
  * a pair a IR sensors, a stopwatch, and the MRD2-S contains relays.  These 
  * units can sense trains, either reflectively (typically `burried' in the 
  * track roadbed) or across the track.  The sensors trigger the stopwatch 
  * and can also trigger the relays.  The SL2 and SR4 modules are intended to
  * operate switch machines, either directly (SL2) or via something like a
  * NCE Switch-It (SR4). Both the SL2 and SR4 also have inputs meant for 
  * local push-button control (but they can be used for other purposes).
  *
  * I was contacted by John Parsons of Azatrax (http://www.azatrax.com) and
  * he kindly sent some samples of the MRD2-U and MRD2-S modules.  This code
  * is the result. The samples allowed me to test the code.
  *
  * @author Robert Heller @<heller\@deepsoft.com@>
  *
  * @{
  */

/**
  * @brief Azatrax C++ LibUSB 1.0 Interface.
  *
  * This is the basic cross-platform class library that uses the libusb-1.0
  * API to communicate with MRD2-U, MRD2-S, SL2, and SR4 modules from Azatrax 
  * over the USB bus.  This library contains classes that encapsulate logic to
  * communicate with these devices.  Each class instance connects to a specific
  * device instance.
  *
  * @author Robert Heller @<heller\@deepsoft.com@>
  *
  * @section azatrax_tcl_package Tcl Package Provided
  *
  * Azatrax 1.0.0
  *
  * @section azatrax_binary_library Library Provided
  *
  * libAzatrax 1.0.0
  */

namespace azatrax {

#ifdef SWIG
%nodefaultctor Azatrax;
#endif


#ifndef SWIG
#define ErrorCode int
class MRD;
class SL2;
class SR4;
#endif

/** @brief Azatrax I/O Class
  *
  * Base Azatrax interface class.  From this base class, a class is derived that
  * implements the device-specific functions.  The base class only implements
  * the shared core functions.  It also includes a static method that implements
  * a factor that creates device instances from device serial numbers.
  *
  * There are two static methods, one that return the number of open devices
  * (active instances) and another that returns a list of discovered serial
  * numbers.
  *
  * @author Robert Heller \<heller\@deepsoft.com\>
  */
  
class Azatrax {
#ifndef SWIGTCL8
protected:
	/** Base constructor.
	  *
	  * @param serialnumber The serial number of the device to open.
	  * @param idProduct The product Id to look for.
	  * @param outmessage To hold an error message, if any.
	  */
	Azatrax(const char *serialnumber, unsigned short int idProduct,
		char **outmessage=NULL);
#endif	
public:
	/** Base destructor.
	  */
	~Azatrax();
	/** @brief Open device count.
	  *
	  * Return the number of open devices.
	  */
	static int NumberOfOpenDevices() {return deviceOpenCount;}
	/** @brief List serial numbers of all connected devices.
	  *
	  * Return a vector of serial number strings.
	  */
	static char ** AllConnectedDevices();
#ifndef SWIGTCL8
	/** @brief Open an Azatrax device by serial number.
	  *
	  * @param serialnumber the serial number to open.
	  * @param idProduct The product id code identifing the type of 
	  *	device to open.  Passing 0 means open any Azatrax device.
	  * @param outmessage To hold an error message, if any.
	  * 
	  */
	static Azatrax *OpenDevice(const char *serialnumber, 
				unsigned short int idProduct=0,
				char **outmessage=NULL);
#endif
#ifndef SWIGTCL8
	friend class MRD;
	friend class SL2;
	friend class SR4;
	/** Command codes. */
	enum commands {
		/** Set channel 1 (MRD2 only) */
		cmd_SetChan1=0x31,
		/** Set channel 2 (MRD2 only) */
		cmd_SetChan2=0x32,
		/** Clear Externally Changed (MRD2 only) */
		cmd_ClearExternallyChanged=0x34,
		/** Disable External Changes (MRD2 only) */
		cmd_DisableExternal=0x37,
		/** Enable External Changes (MRD2 only) */
		cmd_EnableExternal=0x38,
		/** (SL2 only) Sets output terminal Q1 to positive, Q2 to negative. */
		cmd_Q1posQ2neg=0x39,
		/** (SL2 only) Sets output terminal Q1 to negative, Q2 to positive. */
		cmd_Q1negQ2pos=0x3A,
		/** (SL2 only) Outputs Q1 & Q2 both set to open circuit (disconnects switch machine #1) */
		cmd_Q1Q2open=0x3F,
		/** (SL2 only) Sets output terminal Q3 to positive, Q4 to negative. */
		cmd_Q3posQ4neg=0x49,
		/** (SL2 only) Sets output terminal Q3 to negative, Q4 to positive. */
		cmd_Q3negQ4pos=0x4A,
		/** (SL2 only) Outputs Q3 & Q4 both set to open circuit (disconnects switch machine #1) */
		cmd_Q3Q4open=0x4F,
		/** Restore LED Function (All devices) */
		cmd_RestoreLEDFunction=0x50,
		/** Identify LED 1 (All devices) */
		cmd_Identify_1=0x51,
		/** Identify LED 2 (MRD2 only) */
		cmd_Identify_2=0x52,
		/** Identify both LEDs (MRD2 only) */
		cmd_Identify_1_2=0x53,
		/** Reset Stopwatch (MRD2 only) */
		cmd_ResetStopwatch=0x54,
		/**  Get State Data (All devices) */
		cmd_GetStateData=0x57,
		/** 3 bytes - Sets output relay contacts to blinking state.
		  * 2nd Byte is a bit map of the outputs to be affected:
		  * bits 7-4: ignored
		  * bit 3: 1 if Q4 affected
		  * bit 2: 1 if Q3 affected
		  * bit 1: 1 if Q2 affected
		  * bit 0: 1 if Q1 affected
		  * 3rd Byte is the output on/off rate (50% duty cycle):
		  * bits<7:2>: ignored
		  * bits<1:0>: 1,1 - one cycle / 2 seconds
		  *          1,0 - one cycle / second
		  *          0,1 - two cycles / second
		  *          0,0 - four cycles / second
		  */
		cmd_OutputRelayBlink=0x61,
		/** 2 bytes - Sets output relay contacts to 'off' state.
		  * 2nd Byte is a bit map of the outputs to be affected:
		  * bits 7-4: ignored
		  * bit 3: 1 if Q4 affected
		  * bit 2: 1 if Q3 affected
		  * bit 1: 1 if Q2 affected
		  * bit 0: 1 if Q1 affected
		 */
		cmd_OutputRelayOff=0x67,
		/** (SR4 only) 2 bytes - Sets output relay contacts to 'on' state.
		  * 2nd Byte is a bit map of the outputs to be affected:
		  * bits 7-4: ignored
		  * bit 3: 1 if Q4 affected
		  * bit 2: 1 if Q3 affected
		  * bit 1: 1 if Q2 affected
		  * bit 0: 1 if Q1 affected
		  */
		cmd_OutputRelayOn=0x6B,
		/** (SR4 only) 3 bytes - Pulses output relay contacts to momentary 'on' state.
		  * 2nd Byte is a bit map of the outputs to be affected:
		  * bits 7-4: ignored
		  * bit 3: 1 if Q4 affected
		  * bit 2: 1 if Q3 affected
		  * bit 1: 1 if Q2 affected
		  * bit 0: 1 if Q1 affected
		  * 3rd Byte is pulse duration in 0.5 second increments:
		  * 0x00: no pulse generated
		  * 0x01: 0.5 sec
		  * 0x02: 1.0 sec
		  * ...   ...
		  * 0xFE 63.0 sec
		  * 0xFF 63.5 sec
		  */
		cmd_OutputRelayPulse=0x6D,
		/** (SR4 and SL2 only) 2 bytes - Enables/disables discrete input lines from controlling outputs
		  * For SL2:
		  * When enabled, I1 & I2 affect Q1 & Q2 (switch 1), I3 & I4 affect Q3 & Q4
		  * (switch 2).
		  * 2nd Byte is a bit map of the outputs to be affected:
		  * bits 7-4: ignored
		  * bit 3: 1 enables I4, 0 disables
		  * bit 2: 1 enables I3, 0 disables
		  * bit 1: 1 enables I2, 0 disables
		  * bit 0: 1 enables I1, 0 disables
		  * For SR4:
		  * 2nd Byte is a bit map of the inputs:
		  * bits 7-4: ignored
		  * bit 3: 1 if I4 active causes Q4 on, else 0 (I4 does not affect Q4)
		  * bit 2: 1 if I3 active causes Q3 on, else 0 (I3 does not affect Q3)
		  * bit 1: 1 if I2 active causes Q2 on, else 0 (I2 does not affect Q2)
		  * bit 0: 1 if I1 active causes Q1 on, else 0 (I1 does not affect Q1)
		  */
		cmd_OutputRelayInputControl=0x6E
		
	};
	/** Device connection list constants. */
	enum DeviceConnectionList {
		_InitSize = 10,
		_GrowSize = 10
	};
#endif
	/** @brief  Azatrax vendor and product codes.
	  *
	  *  This is the vendor and product codes (Azatrax)
	  * was granted for their USB products.
	  */
	enum {
		/** Azatrax vendor id. */
		idAzatraxVendor = 0x265F,
		/** MRD2 Product id. */
		idMRDProduct = 0xfcb2,
		/** SL2 Product id. */
		idSL2Product = 0xfca1,
		/** SR4 Product id. */
		idSR4Product = 0xfca2 
	};
	/** @brief Restore LED function.
	  *
	  * Restore LED function - On-board LEDs return to their normal
	  * function of indicating status of sensors 1 and 2.
	  */
	ErrorCode RestoreLEDFunction() const {return sendByte(cmd_RestoreLEDFunction);}
	/** @brief Identify 1.
	  *
	  * Identify 1 - Flashes sensor 1's LED.
	  */
	ErrorCode Identify_1() const {return sendByte(cmd_Identify_1);}
	/** @brief Get State Data.
	  *
	  * Get State Data - retrieve 16 byte data packet from sensor to
	  * in memory copy.
	  */
	ErrorCode GetStateData();
	/** Packet count. */
	uint8_t PacketCount() const {return stateDataPacket.packetCount;}
	/** @brief Serial Number.
	  *
	  * Return our serial number.
	  */
	const char *SerialNumber() const {return mySerialNumber;}
	/** @brief My product name.
	  *
	  * Return our Product name.
	  */
	const char *MyProduct() const {
		switch (myProductId) {
			case idMRDProduct: return "MRD";
			case idSL2Product: return "SL2";
			case idSR4Product: return "SR4";
			default: return "unknown";
		}
		return "unknown";
	}
	/** @brief My product code.
	  *
	  * Return our Product Id.
	  */
	unsigned short int MyProductId() const {return myProductId;}
	/** @brief Product id code.
	  *
	  * Return a product ID code.
	  *
	  * @param productName The name of the product.
	  */
	static unsigned short int ProductIdCode(const char *productName) {
		if (strncasecmp(productName,"MRD2",strlen(productName)) == 0) return idMRDProduct;
		else if (strncasecmp(productName,"SL2",strlen(productName)) == 0) return idSL2Product;
		else if (strncasecmp(productName,"SR4",strlen(productName)) == 0) return idSR4Product;
		else return 0;
	}
#ifndef SWIGTCL8
protected:
	/** @brief Is this the device we want?
	  *
	  * Check to see if this device is a Azatrax device and if it has
	  * the serial number we want.
	  *
	  * @param dev libusb_device struct pointer
	  *	(from libusb_get_device_list).
	  * @param serialnumber the serial number we are looking for.
	  * @param idProduct the product id code.
	  */
	static bool IsThisTheAzatraxWeAreLookingFor(libusb_device *dev,
					const char *serialnumber,
					unsigned short int idProduct);
	/** @brief Return the matching product id for this device and serial number.
	  *
	  * Checks if the dev is a Azatrax device, with the specified serial number,
	  * matching for product id.
	  *
	  * @param dev libusb_device struct pointer
	  *	(from libusb_get_device_list).
	  * @param serialnumber the serial number we are looking for.
	  * @param idProductMatch either an exact product id or 0 (means
	  *	any product id)
	  * @return The actual product id (device matched) or 0 (no match).
	  */
	static unsigned short int GetProductId(libusb_device *dev,
					    const char *serialnumber,
					    unsigned short int idProductMatch=0
						);
	/** @brief Send a command byte.
	  *
	  * Sends one command byte on endpoint 01.
	  * @param commandByte the command byte to send.
	  */
	ErrorCode sendByte(uint8_t commandByte) const;
	/** @brief Send a command byte with 1 data byte.
	  *
	  * Sends one command byte on endpoint 01, with one data byte.
	  * @param commandByte The command byte to send.
	  * @param byte2 The data byte to send.
	  */
	ErrorCode send2Bytes(uint8_t commandByte,uint8_t byte2) const;
	/** @brief Send a command bytewith 2 data bytes.
	  *
	  * Sends one command byte on endpoint 01, with two data bytes.
	  * @param commandByte The command byte to send.
	  * @param byte2 The first data byte to send.
	  * @param byte3 The second data byte to send.
	  */
	ErrorCode send3Bytes(uint8_t commandByte,uint8_t byte2,uint8_t byte3) const;
	/** @brief Device open count.
	  *
	  * This is used to decide if and when to call libusb_init
	  * and libusb_exit.
	  */
	static int deviceOpenCount;
	/** @brief USB Device Handle.
	  *
	  *  This is the USB device handle for the
	  * device.
	  */
	struct libusb_device_handle *handle;
	/** @brief Serial number buffer.
	  *
	  * This holds the serial number of the device.
	  */
	char mySerialNumber[10];
	/** @brief Product id. */
	unsigned short int myProductId;
	/** @brief Raw USB Data Packet.
	  *
	  * This is the USB Data Packet returned in response
	  * to the GetStateData command.
	  */
	struct StateDataPacket {
		/** Command Echo byte. (All Devices) */
		uint8_t commandEcho;		// byte 0
		/** Packet counter (All Devices) */
		uint8_t packetCount;		// byte 1
		/** Status byte 1 (All Devices) 
		  * @sa MRD, SL2, and SR4 for detailed bitfields of this byte.
		  */
		uint8_t status1;		// byte 2
		/** Status byte 2 (All Devices) 
		  * @sa MRD, SL2, and SR4 for detailed bitfields of this byte.
		  */
		uint8_t status2;		// byte 3
		/** Status byte 3 (SR4-U, SL2-U) / Stopwatch 1/100ths of a second (MRD2 only) */
		uint8_t status3;		// byte 4
/* Alternitive element name */
#define stopwatchFract status3 
		/** Status byte 4 (SR4-U, SL2-U) / Stopwatch seconds (MRD2 only) */
		uint8_t status4;	// byte 5
/* Alternitive element name */
#define stopwatchSeconds status4
		/** Stopwatch minutes (MRD2 only) */
		uint8_t stopwatchMinutes;	// byte 6
		/** Stopwatch hours (MRD2 only) */
		uint8_t stopwatchHours;		// byte 7
		/** Operating mode (MRD2 only) */
		uint8_t operatingMode;		// byte 8
		/** Reserved bytes */
		uint8_t reserved[6];		// bytes 9-14
		/** End of data */
		uint8_t endOfData;		// byte 15
	} stateDataPacket;

#endif
};
};

#ifdef SWIGTCL8
%apply int MyTcl_Result { int Azatrax_OpenDevice };
int Azatrax_OpenDevice(Tcl_Interp *interp,
                          const char *serialnumber,
                          unsigned short int idProduct);
%{
static int Azatrax_OpenDevice(Tcl_Interp *interp,
			  const char *serialnumber, 
			  unsigned short int idProduct=0)
{
	char ** message = new char*;
	*message = NULL;
	Azatrax *result = Azatrax::OpenDevice(serialnumber,idProduct,message);
#ifdef DEBUG
	fprintf(stderr,"Azatrax_OpenDevice(): result = %llx\n",(long long)result);
#endif
	if (result == NULL) {
		int mlen = strlen(*message);
#ifdef DEBUG
		fprintf(stderr,"Azatrax_OpenDevice(): mlen = %d, *message is '%s'\n",mlen,*message);
#endif
		Tcl_SetObjResult(interp,Tcl_NewStringObj(*message,mlen));
		delete *message;
		delete message;
		return TCL_ERROR;
	}
	delete message;
	Tcl_Obj *resultObj = NULL;
	switch (result->MyProductId()) {
		case Azatrax::idMRDProduct:
			resultObj = SWIG_NewInstanceObj((void *)result,SWIGTYPE_p_azatrax__MRD,0);
			break;
		case Azatrax::idSL2Product:
			resultObj = SWIG_NewInstanceObj((void *)result,SWIGTYPE_p_azatrax__SL2,0);
			break;
		case Azatrax::idSR4Product:
			resultObj = SWIG_NewInstanceObj((void *)result,SWIGTYPE_p_azatrax__SR4,0);
			break;
	}
#ifdef DEBUG
	fprintf(stderr,"Azatrax_OpenDevice(): resultObj is %s\n",Tcl_GetString(resultObj));
#endif
	Tcl_SetObjResult(interp,resultObj);
	return TCL_OK;

}			
%}
#endif


/** @} */

#endif // _AZATRAX_H_

